# EM4x70 Logging

Full log of all bits sent, and all bits received, when
exercising each of the `lf em 4x70` commands.

This can be used to ensure no regressions in what is sent
(and received) when modifying the `lf em 4x70` code.

## Discovered Potential Bugs

* [X] Last four bits of FRN appear to always be 0b1111 (0xF) instead of intended value 0b1100 (0xC)?
      FIXED -- log buffer was too small ... needs to be 98 bits due to inclusion of two RM bits
* [ ] Semi-randomized application of parity to command:
      * EM4X70_COMMAND_ID    -- No parity
      * EM4X70_COMMAND_UM1   -- No parity
      * EM4X70_COMMAND_UM2   -- No parity
      * EM4X70_COMMAND_WRITE -- Always with fifth parity bit added
      * EM4X70_COMMAND_AUTH  -- Always with fifth parity bit added
      * EM4X70_COMMAND_PIN   -- Always with fifth parity bit added

## Comparison of cmds send with / without `--par`

This section will only list the commands sent, and break them down into their components.
It's intended to be compact, and document what the code did as of 2024-05-01 (or thereabouts).
First will be log without, followed by log with `--par` option.

### `lf em 4x70 info`

Bits sent to the tag are IDENTICAL.
The `--par` option is IGNORED ... would have expected
to see CMD values of `0011`, `0101`, and `1111` instead!

```
[#] REM               | RM | CMD    | Addr   | Data....
[#] REM --------------|----|--------|--------|--------------
[#] sent >>>:  6 bits:| 00 | 0001   |        |
[#] sent >>>:  6 bits:| 00 | 0010   |        |
[#] sent >>>:  6 bits:| 00 | 0111   |        |
[#] REM --------------|----|--------|--------|--------------
[#] sent >>>:  6 bits:| 00 | 0001   |        |
[#] sent >>>:  6 bits:| 00 | 0010   |        |
[#] sent >>>:  6 bits:| 00 | 0111   |        |
[#] REM --------------|----|--------|--------|--------------
```


### `lf em 4x70 write -b 13 -d C65B`

When `--par` is used, the command is treated as a three-bit
command, and a parity bit calculated and added.

As can be seen, the existing code was re-using that logic
also for the address and data bits.  This might be OK as
to the address bits (unlikely, but possible).  This is most
definitely wrong as to the data bits.

```
[#] REM               | RM | CMD    | Addr   | Data....
[#] REM --------------|----|--------|--------|--------------
[#] sent >>>: 37 bits:| 00 | 0101 0 | 1101 1 | 1100 0 0110 0 0101 0 1011 1  0100 0
[#] REM --------------|----|--------|--------|--------------
[#] sent >>>: 30 bits:| 00 |  101 0 |  101 0 |  100 1  110 0  101 0  011 0   100 0
[#] REM --------------|----|--------|--------|--------------
```


### `lf em 4x70 setpin --pin 12345678`

```
[#] REM               | RM | CMD    | Addr   | Data....
[#] REM --------------|----|--------|--------|--------------
[#] sent >>>:  6 bits:| 00 | 0001   |        |
[#] sent >>>: 37 bits:| 00 | 0101 0 | 1011 1 | 0001 1 0010 1 0011 0 0100 1  0100 0
[#] sent >>>: 37 bits:| 00 | 0101 0 | 1010 0 | 0101 0 0110 0 0111 1 1000 1  1100 0
[#] sent >>>: 71 bits:| 00 | 0100 1 |        | 0111   1000   1011   1000  // ID  nibbles 1-4
                      |    |        |        | 1110   0000   0001   0010  // ID  nibbles 5-8
                      |    |        |        | 0001   0010   0011   0100  // Pin nibbles 1-4
                      |    |        |        | 0101   0110   0111   1000  // Pin nibbles 5-8
[#] sent >>>:  6 bits:| 00 | 0010   |        |
[#] sent >>>:  6 bits:| 00 | 0111   |        |
[#] REM --------------|----|--------|--------|--------------
[#] sent >>>: 6 bits: | 00 |  001 1 |        |
[#] REM --------------|----|--------|--------|--------------
```

Surprisingly, `setpin` attempts to send `EM4X70_COMMAND_ID` as a 3-bit
value (+ parity over those three bits).  My tag rejects the command
at this point, as it interprets the command as `EM4X70_COMMAND_AUTH`.

TODO: Test code path on tag that requires 3-bit commands w/ parity bit.


### `lf em 4x70 unlock --pin AAAAAAAA`

TODO: Test code path on tag that requires 3-bit commands w/ parity bit.

My current tags properly reject the earliest command (sent as 3-bit
`EM4X70_COMMAND_ID` + its parity) as being `EM4X70_COMMAND_AUTH`.
Thus, it properly rejects the command.

Writes pin to blocks 11, 10, then unlocks using that pin code.

```
[#] REM               | RM | CMD    | Addr   | Data....
[#] REM --------------|----|--------|--------|--------------
[#] sent >>>:  6 bits:| 00 | 0001   |        |
[#] sent >>>: 37 bits:| 00 | 0101 0 | 1011 1 | 0001 1 0010 1 0011 0 0100 1  0100 0
[#] sent >>>: 37 bits:| 00 | 0101 0 | 1010 0 | 0101 0 0110 0 0111 1 1000 1  1100 0
[#] sent >>>: 71 bits:| 00 | 0100 1 |        | 0111   1000   1011   1000 
                      |    |        |        | 1110   0000   0001   0010 
                      |    |        |        | 0001   0010   0011   0100 
                      |    |        |        | 0101   0110   0111   1000 
[#] sent >>>:  6 bits:| 00 | 0010   |        |
[#] sent >>>:  6 bits:| 00 | 0111   |        |
[#] REM --------------|----|--------|--------|--------------
[#] sent >>>: 6 bits: | 00 |  001 1 |        |
[#] REM --------------|----|--------|--------|--------------
```

### `lf em 4x70 setkey -k 022A028C02BE000102030405`

TODO: Test code path on tag that requires 3-bit commands w/ parity bit.

My current tags properly reject the earliest command (sent as 3-bit
`EM4X70_COMMAND_ID` + its parity) as being `EM4X70_COMMAND_AUTH`.
Thus, it properly rejects the command.



```
[#] REM               | RM | CMD    | Addr   | Data....
[#] REM --------------|----|--------|--------|--------------
[#] sent >>>:  6 bits:| 00 | 0001   |        |
[#] sent >>>: 37 bits:| 00 | 0101 0 | 1001 0 | 0000 0 0010 1 0010 1 1010 0  1010 0
[#] sent >>>: 37 bits:| 00 | 0101 0 | 1000 1 | 0000 0 0010 1 1000 1 1100 0  0110 0
[#] sent >>>: 37 bits:| 00 | 0101 0 | 0111 1 | 0000 0 0010 1 1011 1 1110 1  0111 0
[#] sent >>>: 37 bits:| 00 | 0101 0 | 0110 0 | 0000 0 0000 0 0000 0 0001 1  0001 0
[#] sent >>>: 37 bits:| 00 | 0101 0 | 0101 0 | 0000 0 0010 1 0000 0 0011 0  0001 0
[#] sent >>>: 37 bits:| 00 | 0101 0 | 0100 1 | 0000 0 0100 1 0000 0 0101 0  0001 0
[#] REM --------------|----|--------|--------|--------------
[#] sent >>>: 6 bits: | 00 |  001 1 |        |
[#] REM --------------|----|--------|--------|--------------
```


### `lf em 4x70 auth --rnd 7D5167003571F8 --frn 982DBCC0`

The only problem is that the final nibble of FRN
is missing its most significant bit.

```
[#] REM               | RM | CMD    | Addr   | Data....
[#] REM --------------|----|--------|--------|--------------
[#] sent >>>: 98 bits:| 00 | 0011 0 |        | 0111 1101 0101 0001   \  
                                             | 0110 0111 0000 0000    \_Rnd
                                             | 0011 0101 0111 0001    / 
                                             | 1111 1000             /  
                                             | 0000000               - Tdiv
                                             | 1001 1000 0010 1101   \_FRN
                                             | 1011 1100 1100        / 
[#] REM --------------|----|--------|--------|--------------
[#] sent >>>: 96 bits:| 00 |  011 0 |        | 0111 1101 0101 0001   \  
                                             | 0110 0111 0000 0000    \_Rnd
                                             | 0011 0101 0111 0001    / 
                                             | 1111 1000             /  
                                             | 0000000               - Tdiv
                                             | 1001 1000 0010 1101   \_FRN
                                             | 1011 1100  100        / 
[#] REM --------------|----|--------|--------|--------------
```

## More Comprehensive logs ... Without `--par` option

<details><summary>Hiding by default as not critical</summary><p/>

### lf em 4x70 info

`[usb|script] pm3 --> lf em 4x70 info`

#### log

```
[#] sent >>>: [    17169 ..    19545 ] (   2376 )  6 bits: 000001
[#] recv <<<: [    25999 ..    38288 ] (  12289 ) 32 bits: 01111000101110001110000000010010
[#] sent >>>: [    45022 ..    47399 ] (   2377 )  6 bits: 000010
[#] recv <<<: [    53862 ..    66151 ] (  12289 ) 32 bits: 10100001110111110101011001111000
[#] sent >>>: [    72886 ..    75260 ] (   2374 )  6 bits: 000111
[#] recv <<<: [    81702 ..   106281 ] (  24579 ) 64 bits: 0001001000110100101010101010101010101010101010101010101010101010
```

#### decoded log

```
[#] sent >>>:  6 bits: 00 0001
                       2-bit RM: 00
                       4-bit CMD: 0001 (EM4X70_COMMAND_ID)
[#] recv <<<: 32 bits: 0111 1000 1011 1000 1110 0000 0001 0010
                          7    8    B    8    E    0    1    2

[#] sent >>>:  6 bits: 00 0010
                       2-bit RM: 00
                       4-bit CMD: 0010 (EM4X70_COMMAND_UM1)
[#] recv <<<: 32 bits: 1010 0001 1101 1111 0101 0110 0111 1000
                          A    1    D    F    5    6    7    8

[#] sent >>>:  6 bits: 00 0111
                       2-bit RM: 00
                       4-bit CMD: 0111 (EM4X70_COMMAND_UM2)
[#] recv <<<: 64 bits: 0001 0010 0011 0100 1010 1010 1010 1010 1010 1010 1010 1010 1010 1010 1010 1010
                          1    2    3    4    A    A    A    A    A    A    A    A    A    A    A    A
```




#### other output

```
[=] --- Tag Information ---------------------------
[=] Block |   data   | info
[=] ------+----------+-----------------------------
[=]  15   |   12 34  |  UM2
[=]  14   |   AA AA  |  UM2
[=]  13   |   AA AA  |  UM2
[=]  12   |   AA AA  |  UM2
[=] ------+----------+-----------------------------
[=]  11   |   -- --  |  PIN write only
[=]  10   |   -- --  |  PIN write only
[=] ------+----------+-----------------------------
[=]   9   |   -- --  |  KEY write only
[=]   8   |   -- --  |  KEY write only
[=]   7   |   -- --  |  KEY write only
[=]   6   |   -- --  |  KEY write only
[=]   5   |   -- --  |  KEY write only
[=]   4   |   -- --  |  KEY write only
[=] ------+----------+-----------------------------
[=]   3   |   78 B8  |  ID
[=]   2   |   E0 12  |  ID
[=] ------+----------+-----------------------------
[=]   1   |   A1 DF  |  UM1
[=]   0   |   56 78  |  UM1
[=] ------+----------+-----------------------------
[=] 
[=] Tag ID:    78 B8 E0 12
[=] Lockbit 0: 0
[=] Lockbit 1: 1
[=] Tag is UNLOCKED.
[=] 
```

### lf em 4x70 write (UM2 block 15)

`[usb|script] pm3 --> lf em 4x70 write -b 15 -d 576B`

#### log

```
[#] sent >>>: [    17163 ..    31600 ] (  14437 ) 37 bits: 0001010111100101001111011001011111110
[#] recv <<<: no data
[#] ... lines that retrieve ID/UM1/UM2 removed ...
```

#### decoded log

```
[#] sent >>>: 37 bits: 00 01010 11110 10100 10100 10100 10100 00000
                       || ||||| ||||| ||||| ||||| ||||| ||||| \\\\\---- 1111 0 == 5th row: column parity + 0
                       || ||||| ||||| ||||| ||||| ||||| \\\\\---------- 1011 1 == 0xB 4th row nibble + row parity
                       || ||||| ||||| ||||| ||||| \\\\\---------------- 0110 0 == 0x6 3rd row nibble + row parity
                       || ||||| ||||| ||||| \\\\\---------------------- 0111 1 == 0x7 2nd row nibble + row parity
                       || ||||| ||||| \\\\\---------------------------- 0101 0 == 0x5 1st row nibble + row parity
                       || ||||| \\\\\---------------------------------- 1111 0 == address to write to + address parity
                       || \\\\\---------------------------------------- 0101 0 == EM4X70_COMMAND_WRITE + command parity (!!!)
                       \\---------------------------------------------- 00     == RM
```



#### other output

```
[=] --- Tag Information ---------------------------
[=] Block |   data   | info
[=] ------+----------+-----------------------------
[=]  15   |   57 6B  |  UM2
[=]  14   |   AA AA  |  UM2
[=]  13   |   AA AA  |  UM2
[=]  12   |   AA AA  |  UM2
[=] ------+----------+-----------------------------
[=]  11   |   -- --  |  PIN write only
[=]  10   |   -- --  |  PIN write only
[=] ------+----------+-----------------------------
[=]   9   |   -- --  |  KEY write only
[=]   8   |   -- --  |  KEY write only
[=]   7   |   -- --  |  KEY write only
[=]   6   |   -- --  |  KEY write only
[=]   5   |   -- --  |  KEY write only
[=]   4   |   -- --  |  KEY write only
[=] ------+----------+-----------------------------
[=]   3   |   78 B8  |  ID
[=]   2   |   E0 12  |  ID
[=] ------+----------+-----------------------------
[=]   1   |   A1 DF  |  UM1
[=]   0   |   56 78  |  UM1
[=] ------+----------+-----------------------------
[=] 
[=] Tag ID:    78 B8 E0 12
[=] Lockbit 0: 0
[=] Lockbit 1: 1
[=] Tag is UNLOCKED.
[=] 
```

### lf em 4x70 setkey (autorecovery test key)

This writes to block 9..4.
`[usb|script] pm3 --> lf em 4x70 setkey -k 022A028C02BE000102030405`

#### log

```
[#] sent >>>: [    17163 ..    19539 ] (   2376 )  6 bits: 000001
[#] recv <<<: [    25993 ..    38282 ] (  12289 ) 32 bits: 01111000101110001110000000010010
[#] sent >>>: [    45015 ..    59453 ] (  14438 ) 37 bits: 0001010100100000000101001011010010100
[#] recv <<<: no data
[#] sent >>>: [   114111 ..   128547 ] (  14436 ) 37 bits: 0001010100010000000101100011100001100
[#] recv <<<: no data
[#] sent >>>: [   183207 ..   197646 ] (  14439 ) 37 bits: 0001010011110000000101101111110101110
[#] recv <<<: no data
[#] sent >>>: [   252303 ..   266738 ] (  14435 ) 37 bits: 0001010011000000000000000000001100010
[#] recv <<<: no data
[#] sent >>>: [   321387 ..   335820 ] (  14433 ) 37 bits: 0001010010100000000101000000011000010
[#] recv <<<: no data
[#] sent >>>: [   390471 ..   404903 ] (  14432 ) 37 bits: 0001010010010000001001000000101000010
[#] recv <<<: no data
[#] sent >>>: [    17157 ..    55333 ] (  38176 ) 98 bits: 00001101110111100100011110001101111111011101100001001011000011000000001001100111011101100011111100
[#] recv <<<: [    61410 ..    69091 ] (   7681 ) 20 bits: 10010001110110101000
```

#### decoded log

```
[usb|script] pm3 --> lf em 4x70 setkey -k 022A 028C 02BE 0001 0203 0405
                                          

[#] sent >>>:  6 bits: 00 0001
                       RM EM4X70_COMMAND_ID
[#] recv <<<: 32 bits: 0111 1000 1011 1000 1110 0000 0001 0010
                          7    8    B    8    E    0    1    2

[#] sent >>>: 37 bits: 00 01010 10010 00000 00101 00101 10100 10100
                       || ||||| ||||| ||||| ||||| ||||| ||||| \\\\\---- 1010 0 == 0xA 5th row: column parity + 0
                       || ||||| ||||| ||||| ||||| ||||| \\\\\---------- 1010 0 == 0xA 4th row nibble + row parity
                       || ||||| ||||| ||||| ||||| \\\\\---------------- 0010 1 == 0x2 3rd row nibble + row parity
                       || ||||| ||||| ||||| \\\\\---------------------- 0010 1 == 0x2 2nd row nibble + row parity
                       || ||||| ||||| \\\\\---------------------------- 0000 0 == 0x0 1st row nibble + row parity
                       || ||||| \\\\\---------------------------------- 1001 0 == 0x9 address + address parity
                       || \\\\\---------------------------------------- 0101 0 == EM4X70_COMMAND_WRITE + parity
                       \\---------------------------------------------- 00     == RM bits

[#] sent >>>: 37 bits: 00 01010 10001 00000 00101 10001 11000 01100
                       || ||||| ||||| ||||| ||||| ||||| ||||| \\\\\---- 0110 0 == 0x6 5th row: column parity + 0
                       || ||||| ||||| ||||| ||||| ||||| \\\\\---------- 1100 0 == 0xC 4th row nibble + row parity
                       || ||||| ||||| ||||| ||||| \\\\\---------------- 1000 1 == 0x8 3rd row nibble + row parity
                       || ||||| ||||| ||||| \\\\\---------------------- 0010 1 == 0x2 2nd row nibble + row parity
                       || ||||| ||||| \\\\\---------------------------- 0000 0 == 0x0 1st row nibble + row parity
                       || ||||| \\\\\---------------------------------- 1000 1 == 0x8 address + address parity
                       || \\\\\---------------------------------------- 0101 0 == EM4X70_COMMAND_WRITE + parity
                       \\---------------------------------------------- 00     == RM bits

[#] sent >>>: 37 bits: 00 01010 01111 00000 00101 10111 11101 01110
                       || ||||| ||||| ||||| ||||| ||||| ||||| \\\\\---- 0111 0 == 0x7 5th row: column parity + 0
                       || ||||| ||||| ||||| ||||| ||||| \\\\\---------- 1110 1 == 0xE 4th row nibble + row parity
                       || ||||| ||||| ||||| ||||| \\\\\---------------- 1011 1 == 0xB 3rd row nibble + row parity
                       || ||||| ||||| ||||| \\\\\---------------------- 0010 1 == 0x2 2nd row nibble + row parity
                       || ||||| ||||| \\\\\---------------------------- 0000 0 == 0x0 1st row nibble + row parity
                       || ||||| \\\\\---------------------------------- 0111 1 == 0x7 address + address parity
                       || \\\\\---------------------------------------- 0101 0 == EM4X70_COMMAND_WRITE + parity
                       \\---------------------------------------------- 00     == RM bits

[#] sent >>>: 37 bits: 00 01010 01100 00000 00000 00000 00011 00010
                       || ||||| ||||| ||||| ||||| ||||| ||||| \\\\\---- 0001 0 == 0x1 5th row: column parity + 0
                       || ||||| ||||| ||||| ||||| ||||| \\\\\---------- 0001 1 == 0x1 4th row nibble + row parity
                       || ||||| ||||| ||||| ||||| \\\\\---------------- 0000 0 == 0x0 3rd row nibble + row parity
                       || ||||| ||||| ||||| \\\\\---------------------- 0000 0 == 0x0 2nd row nibble + row parity
                       || ||||| ||||| \\\\\---------------------------- 0000 0 == 0x0 1st row nibble + row parity
                       || ||||| \\\\\---------------------------------- 0110 0 == 0x6 address + address parity
                       || \\\\\---------------------------------------- 0101 0 == EM4X70_COMMAND_WRITE + parity
                       \\---------------------------------------------- 00     == RM bits

[#] sent >>>: 37 bits: 00 01010 01010 00000 00101 00000 00110 00010
                       || ||||| ||||| ||||| ||||| ||||| ||||| \\\\\---- 0001 0 == 0x1 5th row: column parity + 0
                       || ||||| ||||| ||||| ||||| ||||| \\\\\---------- 0011 0 == 0x3 4th row nibble + row parity
                       || ||||| ||||| ||||| ||||| \\\\\---------------- 0000 0 == 0x0 3rd row nibble + row parity
                       || ||||| ||||| ||||| \\\\\---------------------- 0010 1 == 0x2 2nd row nibble + row parity
                       || ||||| ||||| \\\\\---------------------------- 0000 0 == 0x0 1st row nibble + row parity
                       || ||||| \\\\\---------------------------------- 0101 0 == 0x5 address + address parity
                       || \\\\\---------------------------------------- 0101 0 == EM4X70_COMMAND_WRITE + parity
                       \\---------------------------------------------- 00     == RM bits

[#] sent >>>: 37 bits: 00 01010 01001 00000 01001 00000 01010 00010
                       || ||||| ||||| ||||| ||||| ||||| ||||| \\\\\---- 0001 0 == 0x1 5th row: column parity + 0
                       || ||||| ||||| ||||| ||||| ||||| \\\\\---------- 0101 0 == 0xC 4th row nibble + row parity
                       || ||||| ||||| ||||| ||||| \\\\\---------------- 0000 0 == 0x8 3rd row nibble + row parity
                       || ||||| ||||| ||||| \\\\\---------------------- 0100 1 == 0x2 2nd row nibble + row parity
                       || ||||| ||||| \\\\\---------------------------- 0000 0 == 0x0 1st row nibble + row parity
                       || ||||| \\\\\---------------------------------- 0100 1 == 0x8 address + address parity
                       || \\\\\---------------------------------------- 0101 0 == EM4X70_COMMAND_WRITE + parity
                       \\---------------------------------------------- 00     == RM bits


Equivalent of: lf em 4x70 auth --rnd EF23C6FEEC2586 --frn 99DD8FC0 --> 91DA80

[#] sent >>>: 98 bits: 00 00110  1110 1111 0010 0011 1100 0110 1111 1110 1110 1100 0010 0101 1000 0110   0000000  1001 1001 1101 1101 1000 1111 1100
                       || |||||  |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| ||||   |||||||  \--/ \--/ \--/ \--/ \--/ \--/ \--/---- FRN
                       || |||||  |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| ||||   |||||||    9    9    D    D    8    F    C
                       || |||||  |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| ||||   \\\\\\\---------------------------------------- Tdiv (always seven zero bits)
                       || |||||  \--/ \--/ \--/ \--/ \--/ \--/ \--/ \--/ \--/ \--/ \--/ \--/ \--/ \--/-------------------------------------------------- 56-bits RN
                       || |||||    E    F    2    3    C    6    F    E    E    C    2    5    8    6
                       || \\\\\---------------------------------------- 0011 0 == EM4X70_COMMAND_AUTH + parity
                       \\---------------------------------------------- 00     == RM bits
[#] recv <<<: 20 bits: 1001 0001 1101 1010 1000
                          9    1    D    A    8 ---- GRN 
```


#### other output
```
[=] Writing new key ( ok )
[=] Verifying auth for new key: 022a028c02be000102030405 -->  lf em 4x70 auth --rnd EF23C6FEEC2586 --frn 99DD8FC0 --> 91DA80
[=] Authenticating with new key ( ok )
```

### lf em 4x70 auth

This is authentication using the autorecovery test key.

`[usb|script] pm3 --> lf em 4x70 auth --rnd EF23C6FEEC2586 --frn 99DD8FC0` (log 1)
`[usb|script] pm3 --> lf em 4x70 auth --rnd 8713F4E00B8716 --frn CB8A1EA0` (log 2)

#### log 1
```
[#] sent >>>: [    17169 ..    55350 ] (  38181 ) 98 bits: 00001101110111100100011110001101111111011101100001001011000011000000001001100111011101100011111100
[#] recv <<<: [    61421 ..    69103 ] (   7682 ) 20 bits: 10010001110110101000
```

#### decoded log 1
```
[#] sent >>>: 98 bits: 00 00110  1110 1111 0010 0011 1100 0110 1111 1110 1110 1100 0010 0101 1000 0110   0000000  1001 1001 1101 1101 1000 1111 1100
                       || |||||  |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| ||||   |||||||  \--/ \--/ \--/ \--/ \--/ \--/ \--/---- FRN
                       || |||||  |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| ||||   |||||||    9    9    D    D    8    F    C
                       || |||||  |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| ||||   |||||||
                       || |||||  |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| ||||   \\\\\\\---------------------------------------- Tdiv (7x zero)
                       || |||||  \--/ \--/ \--/ \--/ \--/ \--/ \--/ \--/ \--/ \--/ \--/ \--/ \--/ \--/-------------------------------------------------- 56-bits RN
                       || |||||    E    F    2    3    C    6    F    E    E    C    2    5    8    6
                       || \\\\\---------------------------------------- 0011 0 == EM4X70_COMMAND_AUTH + parity
                       \\---------------------------------------------- 00     == RM bits
[#] recv <<<: 20 bits: 1001 0001 1101 1010 1000
                          9    1    D    A    8 ---- GRN
```

#### log 2
```
[#] sent >>>: [    21002 ..    59178 ] (  38176 ) 98 bits: 00001101000011100010011111101001110000000001011100001110001011000000001100101110001010000111101010
[#] recv <<<: [        1 ..    73322 ] (  73321 ) 20 bits: 11110100100011100001
```

#### decoded lgo 2
```
[#] sent >>>: 98 bits: 00 00110  1000 0111 0001 0011 1111 0100 1110 0000 0000 1011 1000 0111 0001 0110   0000000  1100 1011 1000 1010 0001 1110 1010
                       || |||||  |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| ||||   |||||||  \--/ \--/ \--/ \--/ \--/ \--/ \--/---- FRN
                       || |||||  |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| ||||   |||||||    C    B    8    A    1    E    A
                       || |||||  |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| ||||   |||||||
                       || |||||  |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| |||| ||||   \\\\\\\---------------------------------------- Tdiv (7x zero)
                       || |||||  \--/ \--/ \--/ \--/ \--/ \--/ \--/ \--/ \--/ \--/ \--/ \--/ \--/ \--/-------------------------------------------------- 56-bits RN
                       || |||||    E    F    2    3    C    6    F    E    E    C    2    5    8    6
                       || \\\\\---------------------------------------- 0011 0 == EM4X70_COMMAND_AUTH + parity
                       \\---------------------------------------------- 00     == RM bits


[#] recv <<<: 20 bits: 1111 0100 1000 1110 0001
                          F    4    8    E    1 ---- GRN
```



#### other output

```
[=] Tag Auth Response: 91 DA 80
```

### lf em 4x70 setpin

Set new pin code (writes to blocks 11, 10)

`[usb|script] pm3 --> lf em 4x70 setpin -p 12345678`

#### log

```
[#] sent >>>: [    17162 ..    19540 ] (   2378 )  6 bits: 000001
[#] recv <<<: [    25992 ..    38282 ] (  12290 ) 32 bits: 01111000101110001110000000010010
[#] sent >>>: [    45014 ..    59445 ] (  14431 ) 37 bits: 0001010101110001100101001100100101000
[#] recv <<<: no data
[#] sent >>>: [   114098 ..   128531 ] (  14433 ) 37 bits: 0001010101000101001100011111000111000
[#] recv <<<: no data
[#] sent >>>: [   183182 ..   210838 ] (  27656 ) 71 bits: 00010010111100010111000111000000001001000010010001101000101011001111000
[#] recv <<<: [   263748 ..   276039 ] (  12291 ) 32 bits: 01111000101110001110000000010010
[#] sent >>>: [   282770 ..   285147 ] (   2377 )  6 bits: 000010
[#] recv <<<: [   291612 ..   303901 ] (  12289 ) 32 bits: 00100001110111110101011001111000
[#] sent >>>: [   310634 ..   313007 ] (   2373 )  6 bits: 000111
[#] recv <<<: [   319451 ..   344030 ] (  24579 ) 64 bits: 1010101010101010101010101010101010101010101010101010101010101010
```

#### decoded log

```
[#] sent >>>:  6 bits: 00 0001
                       RM EM4X70_COMMAND_ID
[#] recv <<<: 32 bits: 0111 1000 1011 1000 1110 0000 0001 0010
                          7    8    B    8    E    0    1    2

[#] sent >>>: 37 bits: 00 01010  10111  00011 00101 00110 01001 01000
                       || |||||  |||||  ||||| ||||| ||||| ||||| \\\\\---- 0100 0 == 0x4 5th row: column parity + 0
                       || |||||  |||||  ||||| ||||| ||||| \\\\\---------- 0100 1 == 0x4 4th row nibble + row parity
                       || |||||  |||||  ||||| ||||| \\\\\---------------- 0011 0 == 0x3 3rd row nibble + row parity
                       || |||||  |||||  ||||| \\\\\---------------------- 0010 1 == 0x2 2nd row nibble + row parity
                       || |||||  |||||  \\\\\---------------------------- 0001 1 == 0x1 1st row nibble + row parity
                       || |||||  \\\\\----------------------------------- 1011 1 == 0xB address + address parity
                       || \\\\\------------------------------------------ 0101 0 == EM4X70_COMMAND_WRITE + parity
                       \\------------------------------------------------ 00     == RM bits

[#] sent >>>: 37 bits: 00 01010  10100  01010 01100 01111 10001 11000
                       || |||||  |||||  ||||| ||||| ||||| ||||| \\\\\---- 1100 0 == 0xC 5th row: column parity + 0
                       || |||||  |||||  ||||| ||||| ||||| \\\\\---------- 1000 1 == 0x8 4th row nibble + row parity
                       || |||||  |||||  ||||| ||||| \\\\\---------------- 0111 1 == 0x7 3rd row nibble + row parity
                       || |||||  |||||  ||||| \\\\\---------------------- 0110 0 == 0x6 2nd row nibble + row parity
                       || |||||  |||||  \\\\\---------------------------- 0101 0 == 0x5 1st row nibble + row parity
                       || |||||  \\\\\----------------------------------- 1010 0 == 0xA address + address parity
                       || \\\\\------------------------------------------ 0101 0 == EM4X70_COMMAND_WRITE + parity
                       \\------------------------------------------------ 00     == RM bits

[#] sent >>>: 71 bits: 00 01001  0111 1000 1011 1000 1110 0000 0001 0010  0001 0010 0011 0100 0101 0110 0111 1000
                       || |||||  |||| |||| |||| |||| |||| |||| |||| ||||  \||/ \||/ \||/ \||/ \||/ \||/ \||/ \||/-- PIN
                       || |||||  |||| |||| |||| |||| |||| |||| |||| ||||    1    2    3    4    5    6    7    8
                       || |||||  \||/ \||/ \||/ \||/ \||/ \||/ \||/ \||/------------------------------------------- Tag ID
                       || |||||    7    8    B    8    E    0    1    2
                       || \\\\\------------------------------------------------------------------------------------ EM4X70_COMMAND_PIN + parity
                       \\------------------------------------------------------------------------------------------ RM bits
[#] recv <<<: 32 bits: 0111 1000 1011 1000 1110 0000 0001 0010    Tag ID
                          7    8    B    8    E    0    1    2

(reads the UM1 and UM2 blocks again ... skipped for brevity)
```


#### other output

```
[=] --- Tag Information ---------------------------
[=] Block |   data   | info
[=] ------+----------+-----------------------------
[=]  15   |   AA AA  |  UM2
[=]  14   |   AA AA  |  UM2
[=]  13   |   AA AA  |  UM2
[=]  12   |   AA AA  |  UM2
[=] ------+----------+-----------------------------
[=]  11   |   -- --  |  PIN write only
[=]  10   |   -- --  |  PIN write only
[=] ------+----------+-----------------------------
[=]   9   |   -- --  |  KEY write only
[=]   8   |   -- --  |  KEY write only
[=]   7   |   -- --  |  KEY write only
[=]   6   |   -- --  |  KEY write only
[=]   5   |   -- --  |  KEY write only
[=]   4   |   -- --  |  KEY write only
[=] ------+----------+-----------------------------
[=]   3   |   78 B8  |  ID
[=]   2   |   E0 12  |  ID
[=] ------+----------+-----------------------------
[=]   1   |   21 DF  |  UM1
[=]   0   |   56 78  |  UM1
[=] ------+----------+-----------------------------
[=] 
[=] Tag ID:    78 B8 E0 12
[=] Lockbit 0: 0
[=] Lockbit 1: 0
[=] Tag is UNLOCKED.
[=] 
[=] Writing new PIN ( ok )
```

### lf em 4x70 write (lock the tag)

This locks the tag by setting top two bits of UM1.

`[usb|script] pm3 --> lf em 4x70 write -b 1 -d E1DF`

#### log

```
[#] sent >>>: [    17164 ..    31601 ] (  14437 ) 37 bits: 0001010000111110100011110111111011010
[#] recv <<<: no data
[#] sent >>>: [    86259 ..    88633 ] (   2374 )  6 bits: 000001
[#] recv <<<: [    95089 ..   107378 ] (  12289 ) 32 bits: 01111000101110001110000000010010
[#] sent >>>: [   114110 ..   116487 ] (   2377 )  6 bits: 000010
[#] recv <<<: [   122952 ..   135240 ] (  12288 ) 32 bits: 11100001110111110101011001111000
[#] sent >>>: [   141975 ..   144351 ] (   2376 )  6 bits: 000111
[#] recv <<<: [   150792 ..   175370 ] (  24578 ) 64 bits: 1010101010101010101010101010101010101010101010101010101010101010
```

#### decoded log

```
[#] sent >>>: 37 bits: 00 01010  00011  11101 00011 11011 11110 11010
                       || |||||  |||||  ||||| ||||| ||||| ||||| \\\\\---- 1101 0 == 0xD 5th row: column parity + 0
                       || |||||  |||||  ||||| ||||| ||||| \\\\\---------- 1111 0 == 0xF 4th row nibble + row parity
                       || |||||  |||||  ||||| ||||| \\\\\---------------- 1101 1 == 0xD 3rd row nibble + row parity
                       || |||||  |||||  ||||| \\\\\---------------------- 0001 1 == 0x1 2nd row nibble + row parity
                       || |||||  |||||  \\\\\---------------------------- 1110 1 == 0xE 1st row nibble + row parity
                       || |||||  \\\\\----------------------------------- 0001 1 == 0x1 address + address parity
                       || \\\\\------------------------------------------ 0101 0 == EM4X70_COMMAND_WRITE + parity
                       \\------------------------------------------------ 00     == RM bits

Skipping reads of ID, UM1, UM2
```


#### other output

```
[=] --- Tag Information ---------------------------
[=] Block |   data   | info
[=] ------+----------+-----------------------------
[=]  15   |   AA AA  |  UM2
[=]  14   |   AA AA  |  UM2
[=]  13   |   AA AA  |  UM2
[=]  12   |   AA AA  |  UM2
[=] ------+----------+-----------------------------
[=]  11   |   -- --  |  PIN write only
[=]  10   |   -- --  |  PIN write only
[=] ------+----------+-----------------------------
[=]   9   |   -- --  |  KEY write only
[=]   8   |   -- --  |  KEY write only
[=]   7   |   -- --  |  KEY write only
[=]   6   |   -- --  |  KEY write only
[=]   5   |   -- --  |  KEY write only
[=]   4   |   -- --  |  KEY write only
[=] ------+----------+-----------------------------
[=]   3   |   78 B8  |  ID
[=]   2   |   E0 12  |  ID
[=] ------+----------+-----------------------------
[=]   1   |   E1 DF  |  UM1
[=]   0   |   56 78  |  UM1
[=] ------+----------+-----------------------------
[=] 
[=] Tag ID:    78 B8 E0 12
[=] Lockbit 0: 1
[=] Lockbit 1: 1
[=] Tag is LOCKED.
[=] 
```

### lf em 4x70 unlock

`[usb|script] pm3 --> lf em 4x70 unlock -p 12345678`

#### log

```
[#] sent >>>: [    17162 ..    19538 ] (   2376 )  6 bits: 000001
[#] recv <<<: [    25992 ..    38282 ] (  12290 ) 32 bits: 01111000101110001110000000010010
[#] sent >>>: [    45015 ..    72674 ] (  27659 ) 71 bits: 00010010111100010111000111000000001001000010010001101000101011001111000
[#] recv <<<: [   125592 ..   137883 ] (  12291 ) 32 bits: 01111000101110001110000000010010
[#] sent >>>: [   144615 ..   146991 ] (   2376 )  6 bits: 000010
[#] recv <<<: [   153457 ..   165744 ] (  12287 ) 32 bits: 00100001110111110101011001111000
[#] sent >>>: [   172478 ..   174854 ] (   2376 )  6 bits: 000111
[#] recv <<<: [   181296 ..   205874 ] (  24578 ) 64 bits: 1010101010101010101010101010101010101010101010101010101010101010
```

#### other output

```
[=] --- Tag Information ---------------------------
[=] Block |   data   | info
[=] ------+----------+-----------------------------
[=]  15   |   AA AA  |  UM2
[=]  14   |   AA AA  |  UM2
[=]  13   |   AA AA  |  UM2
[=]  12   |   AA AA  |  UM2
[=] ------+----------+-----------------------------
[=]  11   |   -- --  |  PIN write only
[=]  10   |   -- --  |  PIN write only
[=] ------+----------+-----------------------------
[=]   9   |   -- --  |  KEY write only
[=]   8   |   -- --  |  KEY write only
[=]   7   |   -- --  |  KEY write only
[=]   6   |   -- --  |  KEY write only
[=]   5   |   -- --  |  KEY write only
[=]   4   |   -- --  |  KEY write only
[=] ------+----------+-----------------------------
[=]   3   |   78 B8  |  ID
[=]   2   |   E0 12  |  ID
[=] ------+----------+-----------------------------
[=]   1   |   21 DF  |  UM1
[=]   0   |   56 78  |  UM1
[=] ------+----------+-----------------------------
[=] 
[=] Tag ID:    78 B8 E0 12
[=] Lockbit 0: 0
[=] Lockbit 1: 0
[=] Tag is UNLOCKED.
[=] 
```

### lf em 4x70 setpin (revert to AAAAAAAA)

Always leave the tag with pin `AAAAAAAA`.

`[usb|script] pm3 --> lf em 4x70 setpin -p AAAAAAAA`

#### log

```
[#] sent >>>: [    17169 ..    19544 ] (   2375 )  6 bits: 000001
[#] recv <<<: [    25998 ..    38289 ] (  12291 ) 32 bits: 01111000101110001110000000010010
[#] sent >>>: [    45021 ..    59464 ] (  14443 ) 37 bits: 0001010101111010010100101001010000000
[#] recv <<<: no data
[#] sent >>>: [   114117 ..   128558 ] (  14441 ) 37 bits: 0001010101001010010100101001010000000
[#] recv <<<: no data
[#] sent >>>: [   183212 ..   210875 ] (  27663 ) 71 bits: 00010010111100010111000111000000001001010101010101010101010101010101010
[#] recv <<<: [   263791 ..   276080 ] (  12289 ) 32 bits: 01111000101110001110000000010010
[#] sent >>>: [   282813 ..   285188 ] (   2375 )  6 bits: 000010
[#] recv <<<: [   291642 ..   303930 ] (  12288 ) 32 bits: 00100001110111110101011001111000
[#] sent >>>: [   310664 ..   313042 ] (   2378 )  6 bits: 000111
[#] recv <<<: [   319482 ..   344060 ] (  24578 ) 64 bits: 1010101010101010101010101010101010101010101010101010101010101010
```

#### decoded log

```
[#] sent >>>:  6 bits: 00 0001
                       RM EM4X70_COMMAND_ID
[#] recv <<<: 32 bits: 0111 1000 1011 1000 1110 0000 0001 0010
                          7    8    B    8    E    0    1    2

[#] sent >>>: 37 bits: 00 01010  10111  10100 10100 10100 10100 00000
                       || |||||  |||||  ||||| ||||| ||||| ||||| \\\\\---- 0000 0 == 0x0 5th row: column parity + 0
                       || |||||  |||||  ||||| ||||| ||||| \\\\\---------- 1010 0 == 0xA 4th row nibble + row parity
                       || |||||  |||||  ||||| ||||| \\\\\---------------- 1010 0 == 0xA 3rd row nibble + row parity
                       || |||||  |||||  ||||| \\\\\---------------------- 1010 0 == 0xA 2nd row nibble + row parity
                       || |||||  |||||  \\\\\---------------------------- 1010 0 == 0xA 1st row nibble + row parity
                       || |||||  \\\\\----------------------------------- 1011 1 == 0xB address + address parity
                       || \\\\\------------------------------------------ 0101 0 == EM4X70_COMMAND_WRITE + parity
                       \\------------------------------------------------ 00     == RM bits

[#] sent >>>: 37 bits: 00 01010  10100  10100 10100 10100 10100 00000
                       || |||||  |||||  ||||| ||||| ||||| ||||| \\\\\---- 0000 0 == 0x0 5th row: column parity + 0
                       || |||||  |||||  ||||| ||||| ||||| \\\\\---------- 1010 0 == 0xA 4th row nibble + row parity
                       || |||||  |||||  ||||| ||||| \\\\\---------------- 1010 0 == 0xA 3rd row nibble + row parity
                       || |||||  |||||  ||||| \\\\\---------------------- 1010 0 == 0xA 2nd row nibble + row parity
                       || |||||  |||||  \\\\\---------------------------- 1010 0 == 0xA 1st row nibble + row parity
                       || |||||  \\\\\----------------------------------- 1010 0 == 0xA address + address parity
                       || \\\\\------------------------------------------ 0101 0 == EM4X70_COMMAND_WRITE + parity
                       \\------------------------------------------------ 00     == RM bits

[#] sent >>>: 71 bits: 00 01001  0111 1000 1011 1000 1110 0000 0001 0010  1010 1010 1010 1010 1010 1010 1010 1010
                       || |||||  |||| |||| |||| |||| |||| |||| |||| ||||  \||/ \||/ \||/ \||/ \||/ \||/ \||/ \||/-- PIN
                       || |||||  |||| |||| |||| |||| |||| |||| |||| ||||    A    A    A    A    A    A    A    A
                       || |||||  \||/ \||/ \||/ \||/ \||/ \||/ \||/ \||/------------------------------------------- Tag ID
                       || |||||    7    8    B    8    E    0    1    2
                       || \\\\\------------------------------------------------------------------------------------ EM4X70_COMMAND_PIN + parity
                       \\------------------------------------------------------------------------------------------ RM bits
[#] recv <<<: 32 bits: 0111 1000 1011 1000 1110 0000 0001 0010
                          7    8    B    8    E    0    1    2

skipping read of UM1 and UM2
```

#### other output

```
[=] --- Tag Information ---------------------------
[=] Block |   data   | info
[=] ------+----------+-----------------------------
[=]  15   |   AA AA  |  UM2
[=]  14   |   AA AA  |  UM2
[=]  13   |   AA AA  |  UM2
[=]  12   |   AA AA  |  UM2
[=] ------+----------+-----------------------------
[=]  11   |   -- --  |  PIN write only
[=]  10   |   -- --  |  PIN write only
[=] ------+----------+-----------------------------
[=]   9   |   -- --  |  KEY write only
[=]   8   |   -- --  |  KEY write only
[=]   7   |   -- --  |  KEY write only
[=]   6   |   -- --  |  KEY write only
[=]   5   |   -- --  |  KEY write only
[=]   4   |   -- --  |  KEY write only
[=] ------+----------+-----------------------------
[=]   3   |   78 B8  |  ID
[=]   2   |   E0 12  |  ID
[=] ------+----------+-----------------------------
[=]   1   |   21 DF  |  UM1
[=]   0   |   56 78  |  UM1
[=] ------+----------+-----------------------------
[=] 
[=] Tag ID:    78 B8 E0 12
[=] Lockbit 0: 0
[=] Lockbit 1: 0
[=] Tag is UNLOCKED.
[=] 
[=] Writing new PIN ( ok )
```


</details>

## General format of the output:

```
^^^^^------- Actual bits sent
[#] DIRECTN_: [    START ..      END ] ( ELAPSED) ## bits: .....
    ^^^^^^^^      ^^^^^^ ..   ^^^^^^     ^^^^^^   ^^       ^^^^^
    ||||||||      ||||||      ||||||     ||||||   ||       \\\\\------- Actual bits sent or received
    ||||||||      ||||||      ||||||     ||||||   \\------------------- Count of bits sent or received
    ||||||||      ||||||      ||||||     \\\\\\------------------------ (END - START)
    ||||||||      ||||||      \\\\\\----------------------------------- Device tick count after last  bit
    ||||||||      \\\\\\----------------------------------------------- Device tick count at    first bit
    \\\\\\\\----------------------------------------------------------- Direction: sent from reader, or received from tag
```

NOTE: bits sent by reader INCLUDE the 2-bit RM (always `00`)
NOTE: bits received by reader EXCLUDE the 12-bit synchronization header (`111111110000`).


## Initialization of the tag

Script to initialize the tag to known starting state:

<details><summary>Hiding by default as not critical</summary><p/>

```
rem set UM2 blocks to `AAAA`
lf em 4x70 write -b 15 -d AAAA
lf em 4x70 write -b 14 -d AAAA
lf em 4x70 write -b 13 -d AAAA
lf em 4x70 write -b 12 -d AAAA
rem set PIN code to `AAAAAAAA`
lf em 4x70 write -b 11 -d AAAA
lf em 4x70 write -b 10 -d AAAA
rem set KEY to `AAAAAAAAAAAA
lf em 4x70 write -b 9 -d AAAA
lf em 4x70 write -b 8 -d AAAA
lf em 4x70 write -b 7 -d AAAA
lf em 4x70 write -b 6 -d AAAA
lf em 4x70 write -b 5 -d AAAA
lf em 4x70 write -b 4 -d AAAA
rem set ID to `78B8E012`
lf em 4x70 write -b 3 -d 78B8
lf em 4x70 write -b 2 -d E012
rem set UM1 to `21DF5678` (unlocked) ... write block 1 last!
lf em 4x70 write -b 0 -d 5678
lf em 4x70 write -b 1 -d 21DF
```

```
[+] 2024-05-15T18:47:50Z remark: set UM2 blocks to `AAAA`
[#] sent >>>: [    17156 ..    31590 ] (  14434 ) 37 bits: 0001010111101010010100101001010000000
[#] recv <<<: no data
[#] sent >>>: [    86240 ..    88618 ] (   2378 )  6 bits: 000001
[#] recv <<<: [    95070 ..   107359 ] (  12289 ) 32 bits: 01111000101110001110000000010010
[#] sent >>>: [   114092 ..   116470 ] (   2378 )  6 bits: 000010
[#] recv <<<: [   122934 ..   135223 ] (  12289 ) 32 bits: 00100001110111110101011001111000
[#] sent >>>: [   141957 ..   144333 ] (   2376 )  6 bits: 000111
[#] recv <<<: [   150774 ..   175352 ] (  24578 ) 64 bits: 1010101010101010101010101010101010101010101010101010101010101010

[=] --- Tag Information ---------------------------
[=] Block |   data   | info
[=] ------+----------+-----------------------------
[=]  15   |   AA AA  |  UM2
[=]  14   |   AA AA  |  UM2
[=]  13   |   AA AA  |  UM2
[=]  12   |   AA AA  |  UM2
[=] ------+----------+-----------------------------
[=]  11   |   -- --  |  PIN write only
[=]  10   |   -- --  |  PIN write only
[=] ------+----------+-----------------------------
[=]   9   |   -- --  |  KEY write only
[=]   8   |   -- --  |  KEY write only
[=]   7   |   -- --  |  KEY write only
[=]   6   |   -- --  |  KEY write only
[=]   5   |   -- --  |  KEY write only
[=]   4   |   -- --  |  KEY write only
[=] ------+----------+-----------------------------
[=]   3   |   78 B8  |  ID
[=]   2   |   E0 12  |  ID
[=] ------+----------+-----------------------------
[=]   1   |   21 DF  |  UM1
[=]   0   |   56 78  |  UM1
[=] ------+----------+-----------------------------
[=]
[=] Tag ID:    78 B8 E0 12
[=] Lockbit 0: 0
[=] Lockbit 1: 0
[=] Tag is UNLOCKED.
[=]

[#] sent >>>: [    17163 ..    31601 ] (  14438 ) 37 bits: 0001010111011010010100101001010000000
[#] recv <<<: no data
[#] sent >>>: [    86259 ..    88635 ] (   2376 )  6 bits: 000001
[#] recv <<<: [    95089 ..   107379 ] (  12290 ) 32 bits: 01111000101110001110000000010010
[#] sent >>>: [   114111 ..   116488 ] (   2377 )  6 bits: 000010
[#] recv <<<: [   122953 ..   135241 ] (  12288 ) 32 bits: 00100001110111110101011001111000
[#] sent >>>: [   141976 ..   144349 ] (   2373 )  6 bits: 000111
[#] recv <<<: [   150793 ..   175371 ] (  24578 ) 64 bits: 1010101010101010101010101010101010101010101010101010101010101010

[=] --- Tag Information ---------------------------
[=] Block |   data   | info
[=] ------+----------+-----------------------------
[=]  15   |   AA AA  |  UM2
[=]  14   |   AA AA  |  UM2
[=]  13   |   AA AA  |  UM2
[=]  12   |   AA AA  |  UM2
[=] ------+----------+-----------------------------
[=]  11   |   -- --  |  PIN write only
[=]  10   |   -- --  |  PIN write only
[=] ------+----------+-----------------------------
[=]   9   |   -- --  |  KEY write only
[=]   8   |   -- --  |  KEY write only
[=]   7   |   -- --  |  KEY write only
[=]   6   |   -- --  |  KEY write only
[=]   5   |   -- --  |  KEY write only
[=]   4   |   -- --  |  KEY write only
[=] ------+----------+-----------------------------
[=]   3   |   78 B8  |  ID
[=]   2   |   E0 12  |  ID
[=] ------+----------+-----------------------------
[=]   1   |   21 DF  |  UM1
[=]   0   |   56 78  |  UM1
[=] ------+----------+-----------------------------
[=]
[=] Tag ID:    78 B8 E0 12
[=] Lockbit 0: 0
[=] Lockbit 1: 0
[=] Tag is UNLOCKED.
[=]

[#] sent >>>: [    17157 ..    31602 ] (  14445 ) 37 bits: 0001010110111010010100101001010000000
[#] recv <<<: no data
[#] sent >>>: [    86252 ..    88627 ] (   2375 )  6 bits: 000001
[#] recv <<<: [    95081 ..   107372 ] (  12291 ) 32 bits: 01111000101110001110000000010010
[#] sent >>>: [   114105 ..   116478 ] (   2373 )  6 bits: 000010
[#] recv <<<: [   122934 ..   135222 ] (  12288 ) 32 bits: 00100001110111110101011001111000
[#] sent >>>: [   141956 ..   144332 ] (   2376 )  6 bits: 000111
[#] recv <<<: [   150775 ..   175353 ] (  24578 ) 64 bits: 1010101010101010101010101010101010101010101010101010101010101010

[=] --- Tag Information ---------------------------
[=] Block |   data   | info
[=] ------+----------+-----------------------------
[=]  15   |   AA AA  |  UM2
[=]  14   |   AA AA  |  UM2
[=]  13   |   AA AA  |  UM2
[=]  12   |   AA AA  |  UM2
[=] ------+----------+-----------------------------
[=]  11   |   -- --  |  PIN write only
[=]  10   |   -- --  |  PIN write only
[=] ------+----------+-----------------------------
[=]   9   |   -- --  |  KEY write only
[=]   8   |   -- --  |  KEY write only
[=]   7   |   -- --  |  KEY write only
[=]   6   |   -- --  |  KEY write only
[=]   5   |   -- --  |  KEY write only
[=]   4   |   -- --  |  KEY write only
[=] ------+----------+-----------------------------
[=]   3   |   78 B8  |  ID
[=]   2   |   E0 12  |  ID
[=] ------+----------+-----------------------------
[=]   1   |   21 DF  |  UM1
[=]   0   |   56 78  |  UM1
[=] ------+----------+-----------------------------
[=]
[=] Tag ID:    78 B8 E0 12
[=] Lockbit 0: 0
[=] Lockbit 1: 0
[=] Tag is UNLOCKED.
[=]

[#] sent >>>: [    17157 ..    31592 ] (  14435 ) 37 bits: 0001010110001010010100101001010000000
[#] recv <<<: no data
[#] sent >>>: [    80480 ..    82858 ] (   2378 )  6 bits: 000001
[#] recv <<<: [    89310 ..   101599 ] (  12289 ) 32 bits: 01111000101110001110000000010010
[#] sent >>>: [   108333 ..   110708 ] (   2375 )  6 bits: 000010
[#] recv <<<: [   117161 ..   129450 ] (  12289 ) 32 bits: 00100001110111110101011001111000
[#] sent >>>: [   136185 ..   138561 ] (   2376 )  6 bits: 000111
[#] recv <<<: [   145002 ..   169581 ] (  24579 ) 64 bits: 1010101010101010101010101010101010101010101010101010101010101010

[=] --- Tag Information ---------------------------
[=] Block |   data   | info
[=] ------+----------+-----------------------------
[=]  15   |   AA AA  |  UM2
[=]  14   |   AA AA  |  UM2
[=]  13   |   AA AA  |  UM2
[=]  12   |   AA AA  |  UM2
[=] ------+----------+-----------------------------
[=]  11   |   -- --  |  PIN write only
[=]  10   |   -- --  |  PIN write only
[=] ------+----------+-----------------------------
[=]   9   |   -- --  |  KEY write only
[=]   8   |   -- --  |  KEY write only
[=]   7   |   -- --  |  KEY write only
[=]   6   |   -- --  |  KEY write only
[=]   5   |   -- --  |  KEY write only
[=]   4   |   -- --  |  KEY write only
[=] ------+----------+-----------------------------
[=]   3   |   78 B8  |  ID
[=]   2   |   E0 12  |  ID
[=] ------+----------+-----------------------------
[=]   1   |   21 DF  |  UM1
[=]   0   |   56 78  |  UM1
[=] ------+----------+-----------------------------
[=]
[=] Tag ID:    78 B8 E0 12
[=] Lockbit 0: 0
[=] Lockbit 1: 0
[=] Tag is UNLOCKED.
[=]

[+] 2024-05-15T18:47:50Z remark: set PIN code to `AAAAAAAA`
[#] sent >>>: [    17157 ..    31593 ] (  14436 ) 37 bits: 0001010101111010010100101001010000000
[#] recv <<<: no data
[#] sent >>>: [    86253 ..    88629 ] (   2376 )  6 bits: 000001
[#] recv <<<: [    95081 ..   107372 ] (  12291 ) 32 bits: 01111000101110001110000000010010
[#] sent >>>: [   114104 ..   116480 ] (   2376 )  6 bits: 000010
[#] recv <<<: [   122946 ..   135234 ] (  12288 ) 32 bits: 00100001110111110101011001111000
[#] sent >>>: [   141968 ..   144342 ] (   2374 )  6 bits: 000111
[#] recv <<<: [   150786 ..   175363 ] (  24577 ) 64 bits: 1010101010101010101010101010101010101010101010101010101010101010

[=] --- Tag Information ---------------------------
[=] Block |   data   | info
[=] ------+----------+-----------------------------
[=]  15   |   AA AA  |  UM2
[=]  14   |   AA AA  |  UM2
[=]  13   |   AA AA  |  UM2
[=]  12   |   AA AA  |  UM2
[=] ------+----------+-----------------------------
[=]  11   |   -- --  |  PIN write only
[=]  10   |   -- --  |  PIN write only
[=] ------+----------+-----------------------------
[=]   9   |   -- --  |  KEY write only
[=]   8   |   -- --  |  KEY write only
[=]   7   |   -- --  |  KEY write only
[=]   6   |   -- --  |  KEY write only
[=]   5   |   -- --  |  KEY write only
[=]   4   |   -- --  |  KEY write only
[=] ------+----------+-----------------------------
[=]   3   |   78 B8  |  ID
[=]   2   |   E0 12  |  ID
[=] ------+----------+-----------------------------
[=]   1   |   21 DF  |  UM1
[=]   0   |   56 78  |  UM1
[=] ------+----------+-----------------------------
[=]
[=] Tag ID:    78 B8 E0 12
[=] Lockbit 0: 0
[=] Lockbit 1: 0
[=] Tag is UNLOCKED.
[=]

[#] sent >>>: [    17163 ..    31598 ] (  14435 ) 37 bits: 0001010101001010010100101001010000000
[#] recv <<<: no data
[#] sent >>>: [    80487 ..    82865 ] (   2378 )  6 bits: 000001
[#] recv <<<: [    89315 ..   101606 ] (  12291 ) 32 bits: 01111000101110001110000000010010
[#] sent >>>: [   108339 ..   110713 ] (   2374 )  6 bits: 000010
[#] recv <<<: [   117167 ..   129457 ] (  12290 ) 32 bits: 00100001110111110101011001111000
[#] sent >>>: [   136190 ..   138563 ] (   2373 )  6 bits: 000111
[#] recv <<<: [   145008 ..   169585 ] (  24577 ) 64 bits: 1010101010101010101010101010101010101010101010101010101010101010

[=] --- Tag Information ---------------------------
[=] Block |   data   | info
[=] ------+----------+-----------------------------
[=]  15   |   AA AA  |  UM2
[=]  14   |   AA AA  |  UM2
[=]  13   |   AA AA  |  UM2
[=]  12   |   AA AA  |  UM2
[=] ------+----------+-----------------------------
[=]  11   |   -- --  |  PIN write only
[=]  10   |   -- --  |  PIN write only
[=] ------+----------+-----------------------------
[=]   9   |   -- --  |  KEY write only
[=]   8   |   -- --  |  KEY write only
[=]   7   |   -- --  |  KEY write only
[=]   6   |   -- --  |  KEY write only
[=]   5   |   -- --  |  KEY write only
[=]   4   |   -- --  |  KEY write only
[=] ------+----------+-----------------------------
[=]   3   |   78 B8  |  ID
[=]   2   |   E0 12  |  ID
[=] ------+----------+-----------------------------
[=]   1   |   21 DF  |  UM1
[=]   0   |   56 78  |  UM1
[=] ------+----------+-----------------------------
[=]
[=] Tag ID:    78 B8 E0 12
[=] Lockbit 0: 0
[=] Lockbit 1: 0
[=] Tag is UNLOCKED.
[=]

[+] 2024-05-15T18:47:51Z remark: set KEY to `AAAAAAAAAAAA
[#] sent >>>: [    17156 ..    31589 ] (  14433 ) 37 bits: 0001010100101010010100101001010000000
[#] recv <<<: no data
[#] sent >>>: [    86242 ..    88616 ] (   2374 )  6 bits: 000001
[#] recv <<<: [    95070 ..   107359 ] (  12289 ) 32 bits: 01111000101110001110000000010010
[#] sent >>>: [   114093 ..   116469 ] (   2376 )  6 bits: 000010
[#] recv <<<: [   122923 ..   135211 ] (  12288 ) 32 bits: 00100001110111110101011001111000
[#] sent >>>: [   141944 ..   144317 ] (   2373 )  6 bits: 000111
[#] recv <<<: [   150761 ..   175341 ] (  24580 ) 64 bits: 1010101010101010101010101010101010101010101010101010101010101010

[=] --- Tag Information ---------------------------
[=] Block |   data   | info
[=] ------+----------+-----------------------------
[=]  15   |   AA AA  |  UM2
[=]  14   |   AA AA  |  UM2
[=]  13   |   AA AA  |  UM2
[=]  12   |   AA AA  |  UM2
[=] ------+----------+-----------------------------
[=]  11   |   -- --  |  PIN write only
[=]  10   |   -- --  |  PIN write only
[=] ------+----------+-----------------------------
[=]   9   |   -- --  |  KEY write only
[=]   8   |   -- --  |  KEY write only
[=]   7   |   -- --  |  KEY write only
[=]   6   |   -- --  |  KEY write only
[=]   5   |   -- --  |  KEY write only
[=]   4   |   -- --  |  KEY write only
[=] ------+----------+-----------------------------
[=]   3   |   78 B8  |  ID
[=]   2   |   E0 12  |  ID
[=] ------+----------+-----------------------------
[=]   1   |   21 DF  |  UM1
[=]   0   |   56 78  |  UM1
[=] ------+----------+-----------------------------
[=]
[=] Tag ID:    78 B8 E0 12
[=] Lockbit 0: 0
[=] Lockbit 1: 0
[=] Tag is UNLOCKED.
[=]

[#] sent >>>: [    17157 ..    31588 ] (  14431 ) 37 bits: 0001010100011010010100101001010000000
[#] recv <<<: no data
[#] sent >>>: [    86240 ..    88615 ] (   2375 )  6 bits: 000001
[#] recv <<<: [    95069 ..   107359 ] (  12290 ) 32 bits: 01111000101110001110000000010010
[#] sent >>>: [   114093 ..   116470 ] (   2377 )  6 bits: 000010
[#] recv <<<: [   122933 ..   135223 ] (  12290 ) 32 bits: 00100001110111110101011001111000
[#] sent >>>: [   141957 ..   144330 ] (   2373 )  6 bits: 000111
[#] recv <<<: [   150774 ..   175352 ] (  24578 ) 64 bits: 1010101010101010101010101010101010101010101010101010101010101010

[=] --- Tag Information ---------------------------
[=] Block |   data   | info
[=] ------+----------+-----------------------------
[=]  15   |   AA AA  |  UM2
[=]  14   |   AA AA  |  UM2
[=]  13   |   AA AA  |  UM2
[=]  12   |   AA AA  |  UM2
[=] ------+----------+-----------------------------
[=]  11   |   -- --  |  PIN write only
[=]  10   |   -- --  |  PIN write only
[=] ------+----------+-----------------------------
[=]   9   |   -- --  |  KEY write only
[=]   8   |   -- --  |  KEY write only
[=]   7   |   -- --  |  KEY write only
[=]   6   |   -- --  |  KEY write only
[=]   5   |   -- --  |  KEY write only
[=]   4   |   -- --  |  KEY write only
[=] ------+----------+-----------------------------
[=]   3   |   78 B8  |  ID
[=]   2   |   E0 12  |  ID
[=] ------+----------+-----------------------------
[=]   1   |   21 DF  |  UM1
[=]   0   |   56 78  |  UM1
[=] ------+----------+-----------------------------
[=]
[=] Tag ID:    78 B8 E0 12
[=] Lockbit 0: 0
[=] Lockbit 1: 0
[=] Tag is UNLOCKED.
[=]

[#] sent >>>: [    17157 ..    31596 ] (  14439 ) 37 bits: 0001010011111010010100101001010000000
[#] recv <<<: no data
[#] sent >>>: [    86252 ..    88630 ] (   2378 )  6 bits: 000001
[#] recv <<<: [    95082 ..   107371 ] (  12289 ) 32 bits: 01111000101110001110000000010010
[#] sent >>>: [   114104 ..   116478 ] (   2374 )  6 bits: 000010
[#] recv <<<: [   122933 ..   135223 ] (  12290 ) 32 bits: 00100001110111110101011001111000
[#] sent >>>: [   141956 ..   144332 ] (   2376 )  6 bits: 000111
[#] recv <<<: [   150775 ..   175352 ] (  24577 ) 64 bits: 1010101010101010101010101010101010101010101010101010101010101010

[=] --- Tag Information ---------------------------
[=] Block |   data   | info
[=] ------+----------+-----------------------------
[=]  15   |   AA AA  |  UM2
[=]  14   |   AA AA  |  UM2
[=]  13   |   AA AA  |  UM2
[=]  12   |   AA AA  |  UM2
[=] ------+----------+-----------------------------
[=]  11   |   -- --  |  PIN write only
[=]  10   |   -- --  |  PIN write only
[=] ------+----------+-----------------------------
[=]   9   |   -- --  |  KEY write only
[=]   8   |   -- --  |  KEY write only
[=]   7   |   -- --  |  KEY write only
[=]   6   |   -- --  |  KEY write only
[=]   5   |   -- --  |  KEY write only
[=]   4   |   -- --  |  KEY write only
[=] ------+----------+-----------------------------
[=]   3   |   78 B8  |  ID
[=]   2   |   E0 12  |  ID
[=] ------+----------+-----------------------------
[=]   1   |   21 DF  |  UM1
[=]   0   |   56 78  |  UM1
[=] ------+----------+-----------------------------
[=]
[=] Tag ID:    78 B8 E0 12
[=] Lockbit 0: 0
[=] Lockbit 1: 0
[=] Tag is UNLOCKED.
[=]

[#] sent >>>: [    17164 ..    31603 ] (  14439 ) 37 bits: 0001010011001010010100101001010000000
[#] recv <<<: no data
[#] sent >>>: [    86259 ..    88635 ] (   2376 )  6 bits: 000001
[#] recv <<<: [    95089 ..   107377 ] (  12288 ) 32 bits: 01111000101110001110000000010010
[#] sent >>>: [   114111 ..   116487 ] (   2376 )  6 bits: 000010
[#] recv <<<: [   122941 ..   135229 ] (  12288 ) 32 bits: 00100001110111110101011001111000
[#] sent >>>: [   141963 ..   144339 ] (   2376 )  6 bits: 000111
[#] recv <<<: [   150780 ..   175359 ] (  24579 ) 64 bits: 1010101010101010101010101010101010101010101010101010101010101010

[=] --- Tag Information ---------------------------
[=] Block |   data   | info
[=] ------+----------+-----------------------------
[=]  15   |   AA AA  |  UM2
[=]  14   |   AA AA  |  UM2
[=]  13   |   AA AA  |  UM2
[=]  12   |   AA AA  |  UM2
[=] ------+----------+-----------------------------
[=]  11   |   -- --  |  PIN write only
[=]  10   |   -- --  |  PIN write only
[=] ------+----------+-----------------------------
[=]   9   |   -- --  |  KEY write only
[=]   8   |   -- --  |  KEY write only
[=]   7   |   -- --  |  KEY write only
[=]   6   |   -- --  |  KEY write only
[=]   5   |   -- --  |  KEY write only
[=]   4   |   -- --  |  KEY write only
[=] ------+----------+-----------------------------
[=]   3   |   78 B8  |  ID
[=]   2   |   E0 12  |  ID
[=] ------+----------+-----------------------------
[=]   1   |   21 DF  |  UM1
[=]   0   |   56 78  |  UM1
[=] ------+----------+-----------------------------
[=]
[=] Tag ID:    78 B8 E0 12
[=] Lockbit 0: 0
[=] Lockbit 1: 0
[=] Tag is UNLOCKED.
[=]

[#] sent >>>: [    17163 ..    31599 ] (  14436 ) 37 bits: 0001010010101010010100101001010000000
[#] recv <<<: no data
[#] sent >>>: [    86259 ..    88633 ] (   2374 )  6 bits: 000001
[#] recv <<<: [    95089 ..   107379 ] (  12290 ) 32 bits: 01111000101110001110000000010010
[#] sent >>>: [   114111 ..   116485 ] (   2374 )  6 bits: 000010
[#] recv <<<: [   122941 ..   135229 ] (  12288 ) 32 bits: 00100001110111110101011001111000
[#] sent >>>: [   141963 ..   144337 ] (   2374 )  6 bits: 000111
[#] recv <<<: [   150781 ..   175359 ] (  24578 ) 64 bits: 1010101010101010101010101010101010101010101010101010101010101010

[=] --- Tag Information ---------------------------
[=] Block |   data   | info
[=] ------+----------+-----------------------------
[=]  15   |   AA AA  |  UM2
[=]  14   |   AA AA  |  UM2
[=]  13   |   AA AA  |  UM2
[=]  12   |   AA AA  |  UM2
[=] ------+----------+-----------------------------
[=]  11   |   -- --  |  PIN write only
[=]  10   |   -- --  |  PIN write only
[=] ------+----------+-----------------------------
[=]   9   |   -- --  |  KEY write only
[=]   8   |   -- --  |  KEY write only
[=]   7   |   -- --  |  KEY write only
[=]   6   |   -- --  |  KEY write only
[=]   5   |   -- --  |  KEY write only
[=]   4   |   -- --  |  KEY write only
[=] ------+----------+-----------------------------
[=]   3   |   78 B8  |  ID
[=]   2   |   E0 12  |  ID
[=] ------+----------+-----------------------------
[=]   1   |   21 DF  |  UM1
[=]   0   |   56 78  |  UM1
[=] ------+----------+-----------------------------
[=]
[=] Tag ID:    78 B8 E0 12
[=] Lockbit 0: 0
[=] Lockbit 1: 0
[=] Tag is UNLOCKED.
[=]

[#] sent >>>: [    17158 ..    31597 ] (  14439 ) 37 bits: 0001010010011010010100101001010000000
[#] recv <<<: no data
[#] sent >>>: [    86253 ..    88631 ] (   2378 )  6 bits: 000001
[#] recv <<<: [    95083 ..   107372 ] (  12289 ) 32 bits: 01111000101110001110000000010010
[#] sent >>>: [   114104 ..   116477 ] (   2373 )  6 bits: 000010
[#] recv <<<: [   122935 ..   135223 ] (  12288 ) 32 bits: 00100001110111110101011001111000
[#] sent >>>: [   141956 ..   144332 ] (   2376 )  6 bits: 000111
[#] recv <<<: [   150775 ..   175352 ] (  24577 ) 64 bits: 1010101010101010101010101010101010101010101010101010101010101010

[=] --- Tag Information ---------------------------
[=] Block |   data   | info
[=] ------+----------+-----------------------------
[=]  15   |   AA AA  |  UM2
[=]  14   |   AA AA  |  UM2
[=]  13   |   AA AA  |  UM2
[=]  12   |   AA AA  |  UM2
[=] ------+----------+-----------------------------
[=]  11   |   -- --  |  PIN write only
[=]  10   |   -- --  |  PIN write only
[=] ------+----------+-----------------------------
[=]   9   |   -- --  |  KEY write only
[=]   8   |   -- --  |  KEY write only
[=]   7   |   -- --  |  KEY write only
[=]   6   |   -- --  |  KEY write only
[=]   5   |   -- --  |  KEY write only
[=]   4   |   -- --  |  KEY write only
[=] ------+----------+-----------------------------
[=]   3   |   78 B8  |  ID
[=]   2   |   E0 12  |  ID
[=] ------+----------+-----------------------------
[=]   1   |   21 DF  |  UM1
[=]   0   |   56 78  |  UM1
[=] ------+----------+-----------------------------
[=]
[=] Tag ID:    78 B8 E0 12
[=] Lockbit 0: 0
[=] Lockbit 1: 0
[=] Tag is UNLOCKED.
[=]

[+] 2024-05-15T18:47:52Z remark: set ID to `78B8E012`
[#] sent >>>: [    17163 ..    31609 ] (  14446 ) 37 bits: 0001010001100111110001101111000111000
[#] recv <<<: no data
[#] sent >>>: [    86259 ..    88637 ] (   2378 )  6 bits: 000001
[#] recv <<<: [    95089 ..   107379 ] (  12290 ) 32 bits: 01111000101110001110000000010010
[#] sent >>>: [   114112 ..   116487 ] (   2375 )  6 bits: 000010
[#] recv <<<: [   122940 ..   135230 ] (  12290 ) 32 bits: 00100001110111110101011001111000
[#] sent >>>: [   141963 ..   144336 ] (   2373 )  6 bits: 000111
[#] recv <<<: [   150780 ..   175358 ] (  24578 ) 64 bits: 1010101010101010101010101010101010101010101010101010101010101010

[=] --- Tag Information ---------------------------
[=] Block |   data   | info
[=] ------+----------+-----------------------------
[=]  15   |   AA AA  |  UM2
[=]  14   |   AA AA  |  UM2
[=]  13   |   AA AA  |  UM2
[=]  12   |   AA AA  |  UM2
[=] ------+----------+-----------------------------
[=]  11   |   -- --  |  PIN write only
[=]  10   |   -- --  |  PIN write only
[=] ------+----------+-----------------------------
[=]   9   |   -- --  |  KEY write only
[=]   8   |   -- --  |  KEY write only
[=]   7   |   -- --  |  KEY write only
[=]   6   |   -- --  |  KEY write only
[=]   5   |   -- --  |  KEY write only
[=]   4   |   -- --  |  KEY write only
[=] ------+----------+-----------------------------
[=]   3   |   78 B8  |  ID
[=]   2   |   E0 12  |  ID
[=] ------+----------+-----------------------------
[=]   1   |   21 DF  |  UM1
[=]   0   |   56 78  |  UM1
[=] ------+----------+-----------------------------
[=]
[=] Tag ID:    78 B8 E0 12
[=] Lockbit 0: 0
[=] Lockbit 1: 0
[=] Tag is UNLOCKED.
[=]

[#] sent >>>: [    17163 ..    31604 ] (  14441 ) 37 bits: 0001010001011110100000000110010111010
[#] recv <<<: no data
[#] sent >>>: [    80499 ..    82872 ] (   2373 )  6 bits: 000001
[#] recv <<<: [    89328 ..   101618 ] (  12290 ) 32 bits: 01111000101110001110000000010010
[#] sent >>>: [   108350 ..   110726 ] (   2376 )  6 bits: 000010
[#] recv <<<: [   117192 ..   129481 ] (  12289 ) 32 bits: 00100001110111110101011001111000
[#] sent >>>: [   136216 ..   138590 ] (   2374 )  6 bits: 000111
[#] recv <<<: [   145032 ..   169610 ] (  24578 ) 64 bits: 1010101010101010101010101010101010101010101010101010101010101010

[=] --- Tag Information ---------------------------
[=] Block |   data   | info
[=] ------+----------+-----------------------------
[=]  15   |   AA AA  |  UM2
[=]  14   |   AA AA  |  UM2
[=]  13   |   AA AA  |  UM2
[=]  12   |   AA AA  |  UM2
[=] ------+----------+-----------------------------
[=]  11   |   -- --  |  PIN write only
[=]  10   |   -- --  |  PIN write only
[=] ------+----------+-----------------------------
[=]   9   |   -- --  |  KEY write only
[=]   8   |   -- --  |  KEY write only
[=]   7   |   -- --  |  KEY write only
[=]   6   |   -- --  |  KEY write only
[=]   5   |   -- --  |  KEY write only
[=]   4   |   -- --  |  KEY write only
[=] ------+----------+-----------------------------
[=]   3   |   78 B8  |  ID
[=]   2   |   E0 12  |  ID
[=] ------+----------+-----------------------------
[=]   1   |   21 DF  |  UM1
[=]   0   |   56 78  |  UM1
[=] ------+----------+-----------------------------
[=]
[=] Tag ID:    78 B8 E0 12
[=] Lockbit 0: 0
[=] Lockbit 1: 0
[=] Tag is UNLOCKED.
[=]

[+] 2024-05-15T18:47:52Z remark: set UM1 to `21DF5678` (unlocked) ... write block 1 last!
[#] sent >>>: [    17157 ..    31590 ] (  14433 ) 37 bits: 0001010000000101001100011111000111000
[#] recv <<<: no data
[#] sent >>>: [    80482 ..    82860 ] (   2378 )  6 bits: 000001
[#] recv <<<: [    89310 ..   101601 ] (  12291 ) 32 bits: 01111000101110001110000000010010
[#] sent >>>: [   108333 ..   110709 ] (   2376 )  6 bits: 000010
[#] recv <<<: [   117175 ..   129464 ] (  12289 ) 32 bits: 00100001110111110101011001111000
[#] sent >>>: [   136198 ..   138572 ] (   2374 )  6 bits: 000111
[#] recv <<<: [   145014 ..   169592 ] (  24578 ) 64 bits: 1010101010101010101010101010101010101010101010101010101010101010

[=] --- Tag Information ---------------------------
[=] Block |   data   | info
[=] ------+----------+-----------------------------
[=]  15   |   AA AA  |  UM2
[=]  14   |   AA AA  |  UM2
[=]  13   |   AA AA  |  UM2
[=]  12   |   AA AA  |  UM2
[=] ------+----------+-----------------------------
[=]  11   |   -- --  |  PIN write only
[=]  10   |   -- --  |  PIN write only
[=] ------+----------+-----------------------------
[=]   9   |   -- --  |  KEY write only
[=]   8   |   -- --  |  KEY write only
[=]   7   |   -- --  |  KEY write only
[=]   6   |   -- --  |  KEY write only
[=]   5   |   -- --  |  KEY write only
[=]   4   |   -- --  |  KEY write only
[=] ------+----------+-----------------------------
[=]   3   |   78 B8  |  ID
[=]   2   |   E0 12  |  ID
[=] ------+----------+-----------------------------
[=]   1   |   21 DF  |  UM1
[=]   0   |   56 78  |  UM1
[=] ------+----------+-----------------------------
[=]
[=] Tag ID:    78 B8 E0 12
[=] Lockbit 0: 0
[=] Lockbit 1: 0
[=] Tag is UNLOCKED.
[=]

[#] sent >>>: [    17163 ..    31599 ] (  14436 ) 37 bits: 0001010000110010100011110111111000010
[#] recv <<<: no data
[#] sent >>>: [    80486 ..    82864 ] (   2378 )  6 bits: 000001
[#] recv <<<: [    89316 ..   101606 ] (  12290 ) 32 bits: 01111000101110001110000000010010
[#] sent >>>: [   108338 ..   110714 ] (   2376 )  6 bits: 000010
[#] recv <<<: [   117180 ..   129468 ] (  12288 ) 32 bits: 00100001110111110101011001111000
[#] sent >>>: [   136202 ..   138578 ] (   2376 )  6 bits: 000111
[#] recv <<<: [   145020 ..   169599 ] (  24579 ) 64 bits: 1010101010101010101010101010101010101010101010101010101010101010

[=] --- Tag Information ---------------------------
[=] Block |   data   | info
[=] ------+----------+-----------------------------
[=]  15   |   AA AA  |  UM2
[=]  14   |   AA AA  |  UM2
[=]  13   |   AA AA  |  UM2
[=]  12   |   AA AA  |  UM2
[=] ------+----------+-----------------------------
[=]  11   |   -- --  |  PIN write only
[=]  10   |   -- --  |  PIN write only
[=] ------+----------+-----------------------------
[=]   9   |   -- --  |  KEY write only
[=]   8   |   -- --  |  KEY write only
[=]   7   |   -- --  |  KEY write only
[=]   6   |   -- --  |  KEY write only
[=]   5   |   -- --  |  KEY write only
[=]   4   |   -- --  |  KEY write only
[=] ------+----------+-----------------------------
[=]   3   |   78 B8  |  ID
[=]   2   |   E0 12  |  ID
[=] ------+----------+-----------------------------
[=]   1   |   21 DF  |  UM1
[=]   0   |   56 78  |  UM1
[=] ------+----------+-----------------------------
[=]
[=] Tag ID:    78 B8 E0 12
[=] Lockbit 0: 0
[=] Lockbit 1: 0
[=] Tag is UNLOCKED.
[=]
```

</details>

## End of document
