# (c) Copyright 2009-2022. CodeWeavers, Inc.

from gi.repository import Gtk

import c4profiles
import c4profilesmanager
import cxguitools
import cxproduct
import distversion
import ratingutils

import logindialog

from cxutils import cxgettext as _


class RatingController:

    max_characters = 280

    def __init__(self, bottle, parent, is_nag=False, ask_again=True):
        self.xml = Gtk.Builder()
        self.xml.set_translation_domain('crossover')
        self.xml.add_from_file(cxguitools.get_ui_path('ratingdialog'))
        self.xml.connect_signals(self)

        if not bottle or not bottle.appid:
            return

        if distversion.IS_PREVIEW:
            return

        if is_nag and cxproduct.get_config_value('CrossOver', 'AskForRatings', '1') != '1':
            return

        self.rating = None
        self.bottle = bottle
        self.rating_utils = ratingutils.RatingUtils(self.bottle.name, self.bottle.appid)
        success = self.rating_utils.get_rating()

        profiles = c4profilesmanager.C4ProfilesSet.all_profiles()
        profile = profiles.get('com.codeweavers.c4.%s' % self.bottle.appid)

        # Don't ask for a rating if we can't connect to the API or if
        # the user has already submitted a rating.
        if (not success and self.rating_utils.connection_failed) or (is_nag and self.rating_utils.rating) \
           or (is_nag and profile and profile.is_component):
            if (is_nag and self.rating_utils.rating) or (is_nag and profile and profile.is_component):
                self.bottle.disable_rating_nag()

            return

        name = self.rating_utils.name
        if profile:
            name = profile.name

        self.xml.get_object('RatingDialog').set_title(_("Submit a Rating for '%s'") % name)

        self.xml.get_object('RatingLabel').set_text(
            _("Please leave a rating in order to help us better target our improvements to %(product)s and to inform other users on how well '%(application)s' works with %(product)s.") % {'product': distversion.PRODUCT_NAME, 'application': name})

        if is_nag and ask_again:
            self.xml.get_object('LaterRatingButton').show()
            self.xml.get_object('NeverRatingButton').show()
        elif is_nag:
            self.xml.get_object('SkipRatingButton').show()
        else:
            self.xml.get_object('CancelRatingButton').show()

        if is_nag:
            self.xml.get_object('RatingCheckButton').show()

        self.placeholder_text = _('Additional notes (optional)')
        self.has_placeholder_text = True
        self.xml.get_object('RatingEntry').get_buffer().set_text(self.placeholder_text)
        self.xml.get_object('RatingEntry').get_style_context().add_class('dim-label')
        self.xml.get_object('CharactersLeft').get_style_context().add_class('dim-label')
        self.set_character_length(0)

        self.xml.get_object('RatingEntry').get_buffer().connect('changed', self.on_text_changed)
        self.xml.get_object('RatingEntry').get_buffer().connect_after('insert-text', self.on_text_insert)

        self.update_rating(self.rating_utils.rating)
        if self.rating_utils.notes:
            self.has_placeholder_text = False
            self.xml.get_object('RatingEntry').get_style_context().remove_class('dim-label')
            self.xml.get_object('RatingEntry').get_buffer().set_text(self.rating_utils.notes)

        self.xml.get_object('RatingDialog').set_transient_for(parent)
        self.xml.get_object('RatingDialog').show()

        # Show the rating box manually so it doesn't grab focus
        self.xml.get_object('RatingBox').show()

    def update_rating(self, rating, temp_rating=False):
        if rating and 1 <= rating <= 5:
            icon_names = [cxguitools.get_icon_name(('starred', ), Gtk.IconSize.LARGE_TOOLBAR, symbolic=True)] * rating \
                + [cxguitools.get_icon_name(('non-starred', ), Gtk.IconSize.LARGE_TOOLBAR, symbolic=True)] * (5 - rating)
        else:
            icon_names = [cxguitools.get_icon_name(('non-starred', ), Gtk.IconSize.LARGE_TOOLBAR, symbolic=True)] * 5

        for i, icon in enumerate(icon_names):
            self.xml.get_object('star ' + str(i)).set_from_icon_name(icon, Gtk.IconSize.LARGE_TOOLBAR)

        text = c4profiles.RATING_DESCRIPTIONS.get(rating, '')
        tooltip = self.rating_tooltips.get(rating, '')

        self.xml.get_object('PackageRatingLabel').set_text(text)
        self.xml.get_object('PackageRatingLabel').set_tooltip_text(tooltip)
        self.xml.get_object('PackageRatingView').set_tooltip_text(tooltip)

        if not temp_rating:
            self.rating = rating

    def get_window(self):
        return self.xml.get_object("RatingDialog")

    def set_character_length(self, length):
        if self.has_placeholder_text:
            length = 0

        self.xml.get_object('CharactersLeft').set_text(_('Max %(max)d characters, %(left)d left') % {
            'max': self.max_characters, 'left': self.max_characters - length})

    @staticmethod
    def on_RatingCheckButton_toggled(caller):
        if caller.get_active():
            # Don't ask again
            cxproduct.set_config_value('CrossOver', 'AskForRatings', '0')
        else:
            cxproduct.set_config_value('CrossOver', 'AskForRatings', '1')

    def on_CancelRatingButton_clicked(self, _button):
        dialog = self.xml.get_object('RatingDialog')
        dialog.destroy()

    def on_NeverRatingButton_clicked(self, _button):
        self.bottle.disable_rating_nag()
        dialog = self.xml.get_object('RatingDialog')
        dialog.destroy()

    def on_SubmitRatingButton_clicked(self, _button=None):
        dialog = self.xml.get_object('RatingDialog')
        success = self.rating_utils.set_rating(self.rating, self.get_text())
        if not success and self.rating_utils.login_required:
            logindialog.LoginController(dialog.get_toplevel(), self.on_SubmitRatingButton_clicked)
            return

        if not success:
            cxguitools.CXMessageDlg(primary=_('Rating submission failed'), buttons=Gtk.ButtonsType.OK,
                                    parent=self.xml.get_object('RatingDialog').get_toplevel(),
                                    message_type=Gtk.MessageType.ERROR)
            return

        self.bottle.disable_rating_nag()
        dialog.destroy()

    def on_rating_set(self, widget, _event):
        rating = int(Gtk.Buildable.get_name(widget)[-1]) + 1
        self.update_rating(rating)

    def on_rating_enter(self, widget, _event):
        rating = int(Gtk.Buildable.get_name(widget)[-1]) + 1
        self.update_rating(rating, temp_rating=True)

    def on_rating_leave(self, _widget, _event):
        self.update_rating(self.rating)

    def on_text_focus_in(self, widget, _event):
        if self.has_placeholder_text:
            widget.get_style_context().remove_class('dim-label')
            self.has_placeholder_text = False
            widget.get_buffer().set_text('')

    def on_text_focus_out(self, widget, _event):
        textbuf = widget.get_buffer()
        if textbuf.get_text(textbuf.get_start_iter(), textbuf.get_end_iter(), True) == '':
            widget.get_style_context().add_class('dim-label')
            self.has_placeholder_text = True
            textbuf.set_text(self.placeholder_text)

    def on_text_insert(self, textbuf, location, _text, _length):
        length = textbuf.get_char_count()
        if length > self.max_characters:
            offset = textbuf.get_iter_at_offset(self.max_characters)
            end = textbuf.get_end_iter()
            textbuf.delete(offset, end)
            location.assign(offset)
            length = self.max_characters

        self.set_character_length(length)

    def on_text_changed(self, textbuf):
        length = textbuf.get_char_count()
        self.set_character_length(length)

    def get_text(self):
        if self.has_placeholder_text:
            return ''

        textbuf = self.xml.get_object('RatingEntry').get_buffer()
        start, end = textbuf.get_bounds()
        return textbuf.get_text(start, end, False)

    rating_tooltips = {
        5: _("This application installs and runs great, the same as it would in Windows.\n\nPractically all of the application is functional. Using this application in CrossOver should be the same as running the application on native Windows OS."),
        4: _("This application installs and runs quite well, with only minor glitches found.\n\nThe core functionality of this Windows application works fine under CrossOver. There still may be some extra functions that don't work right, but overall this application should be mostly usable. We don't maintain a list of what specific functions work/don't work per-application in our compatibility database, but you may want to check this application's Forums at the CodeWeavers website and see if other users have reported any particular details about running this Windows application in CrossOver."),
        3: _("This application installs and runs, but has problems.\n\nThis application runs, but there are significant functional gaps. CrossOver translates Windows commands to their corresponding %(platform)s commands and back. When a Windows program uses commands that CrossOver doesn't know how to translate the Windows program may not respond when you click things, have garbled output, or crash. You might be able to add different dependencies or try workarounds to get this application to run better, so please check this application's page at the CodeWeavers website. It's also possible that programming added in a future release of CrossOver might help this Windows application run better, so check back later.") % {'platform': distversion.PLATFORM},
        2: _("This application installs, but fails to run.\n\nThe Windows application immediately closes or freezes when you try to run it. You may be able to add various dependencies or try different workarounds to get this application to start, so please check the Support section of the CodeWeavers website for more information and tutorials. If the program still won't launch then whatever programming this Windows application needs to work doesn't exist yet in CrossOver."),
        1: _("This application won't get through installation.\n\nMost installers are standardized and CrossOver will be able to run them without a problem. However some companies customize the installer, or write their own, and the custom programming fails in CrossOver. The Windows application itself might be functional, but there's no way to get it out of the installer."),
        0: _("No ratings have been reported.\n\nAll application entries start at this rating. Over time staff and users can visit the App Compatibility page of the CodeWeavers website and rate how far they got in running this Windows application in CrossOver. If you try this application please submit a rating for how this application worked, or didn't, in CrossOver."),
        -1: _("This installation recipe is from an unknown source. Installing may pose a security risk to your system."),
    }
