﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/eks/EKSRequest.h>
#include <aws/eks/EKS_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace EKS {
namespace Model {

/**
 */
class ListAccessEntriesRequest : public EKSRequest {
 public:
  AWS_EKS_API ListAccessEntriesRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListAccessEntries"; }

  AWS_EKS_API Aws::String SerializePayload() const override;

  AWS_EKS_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>The name of your cluster.</p>
   */
  inline const Aws::String& GetClusterName() const { return m_clusterName; }
  inline bool ClusterNameHasBeenSet() const { return m_clusterNameHasBeenSet; }
  template <typename ClusterNameT = Aws::String>
  void SetClusterName(ClusterNameT&& value) {
    m_clusterNameHasBeenSet = true;
    m_clusterName = std::forward<ClusterNameT>(value);
  }
  template <typename ClusterNameT = Aws::String>
  ListAccessEntriesRequest& WithClusterName(ClusterNameT&& value) {
    SetClusterName(std::forward<ClusterNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of an <code>AccessPolicy</code>. When you specify an access policy
   * ARN, only the access entries associated to that access policy are returned. For
   * a list of available policy ARNs, use <code>ListAccessPolicies</code>.</p>
   */
  inline const Aws::String& GetAssociatedPolicyArn() const { return m_associatedPolicyArn; }
  inline bool AssociatedPolicyArnHasBeenSet() const { return m_associatedPolicyArnHasBeenSet; }
  template <typename AssociatedPolicyArnT = Aws::String>
  void SetAssociatedPolicyArn(AssociatedPolicyArnT&& value) {
    m_associatedPolicyArnHasBeenSet = true;
    m_associatedPolicyArn = std::forward<AssociatedPolicyArnT>(value);
  }
  template <typename AssociatedPolicyArnT = Aws::String>
  ListAccessEntriesRequest& WithAssociatedPolicyArn(AssociatedPolicyArnT&& value) {
    SetAssociatedPolicyArn(std::forward<AssociatedPolicyArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of results, returned in paginated output. You receive
   * <code>maxResults</code> in a single page, along with a <code>nextToken</code>
   * response element. You can see the remaining results of the initial request by
   * sending another request with the returned <code>nextToken</code> value. This
   * value can be between 1 and 100. If you don't use this parameter, 100 results and
   * a <code>nextToken</code> value, if applicable, are returned.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListAccessEntriesRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The <code>nextToken</code> value returned from a previous paginated request,
   * where <code>maxResults</code> was used and the results exceeded the value of
   * that parameter. Pagination continues from the end of the previous results that
   * returned the <code>nextToken</code> value. This value is null when there are no
   * more results to return.</p>  <p>This token should be treated as an opaque
   * identifier that is used only to retrieve the next items in a list and not for
   * other programmatic purposes.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListAccessEntriesRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_clusterName;
  bool m_clusterNameHasBeenSet = false;

  Aws::String m_associatedPolicyArn;
  bool m_associatedPolicyArnHasBeenSet = false;

  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace EKS
}  // namespace Aws
