﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/lex/LexRuntimeServiceRequest.h>
#include <aws/lex/LexRuntimeService_EXPORTS.h>
#include <aws/lex/model/ActiveContext.h>

#include <utility>

namespace Aws {
namespace LexRuntimeService {
namespace Model {

/**
 */
class PostTextRequest : public LexRuntimeServiceRequest {
 public:
  AWS_LEXRUNTIMESERVICE_API PostTextRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "PostText"; }

  AWS_LEXRUNTIMESERVICE_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The name of the Amazon Lex bot.</p>
   */
  inline const Aws::String& GetBotName() const { return m_botName; }
  inline bool BotNameHasBeenSet() const { return m_botNameHasBeenSet; }
  template <typename BotNameT = Aws::String>
  void SetBotName(BotNameT&& value) {
    m_botNameHasBeenSet = true;
    m_botName = std::forward<BotNameT>(value);
  }
  template <typename BotNameT = Aws::String>
  PostTextRequest& WithBotName(BotNameT&& value) {
    SetBotName(std::forward<BotNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The alias of the Amazon Lex bot.</p>
   */
  inline const Aws::String& GetBotAlias() const { return m_botAlias; }
  inline bool BotAliasHasBeenSet() const { return m_botAliasHasBeenSet; }
  template <typename BotAliasT = Aws::String>
  void SetBotAlias(BotAliasT&& value) {
    m_botAliasHasBeenSet = true;
    m_botAlias = std::forward<BotAliasT>(value);
  }
  template <typename BotAliasT = Aws::String>
  PostTextRequest& WithBotAlias(BotAliasT&& value) {
    SetBotAlias(std::forward<BotAliasT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the client application user. Amazon Lex uses this to identify a
   * user's conversation with your bot. At runtime, each request must contain the
   * <code>userID</code> field.</p> <p>To decide the user ID to use for your
   * application, consider the following factors.</p> <ul> <li> <p>The
   * <code>userID</code> field must not contain any personally identifiable
   * information of the user, for example, name, personal identification numbers, or
   * other end user personal information.</p> </li> <li> <p>If you want a user to
   * start a conversation on one device and continue on another device, use a
   * user-specific identifier.</p> </li> <li> <p>If you want the same user to be able
   * to have two independent conversations on two different devices, choose a
   * device-specific identifier.</p> </li> <li> <p>A user can't have two independent
   * conversations with two different versions of the same bot. For example, a user
   * can't have a conversation with the PROD and BETA versions of the same bot. If
   * you anticipate that a user will need to have conversation with two different
   * versions, for example, while testing, include the bot alias in the user ID to
   * separate the two conversations.</p> </li> </ul>
   */
  inline const Aws::String& GetUserId() const { return m_userId; }
  inline bool UserIdHasBeenSet() const { return m_userIdHasBeenSet; }
  template <typename UserIdT = Aws::String>
  void SetUserId(UserIdT&& value) {
    m_userIdHasBeenSet = true;
    m_userId = std::forward<UserIdT>(value);
  }
  template <typename UserIdT = Aws::String>
  PostTextRequest& WithUserId(UserIdT&& value) {
    SetUserId(std::forward<UserIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Application-specific information passed between Amazon Lex and a client
   * application.</p> <p>For more information, see <a
   * href="https://docs.aws.amazon.com/lex/latest/dg/context-mgmt.html#context-mgmt-session-attribs">Setting
   * Session Attributes</a>.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetSessionAttributes() const { return m_sessionAttributes; }
  inline bool SessionAttributesHasBeenSet() const { return m_sessionAttributesHasBeenSet; }
  template <typename SessionAttributesT = Aws::Map<Aws::String, Aws::String>>
  void SetSessionAttributes(SessionAttributesT&& value) {
    m_sessionAttributesHasBeenSet = true;
    m_sessionAttributes = std::forward<SessionAttributesT>(value);
  }
  template <typename SessionAttributesT = Aws::Map<Aws::String, Aws::String>>
  PostTextRequest& WithSessionAttributes(SessionAttributesT&& value) {
    SetSessionAttributes(std::forward<SessionAttributesT>(value));
    return *this;
  }
  template <typename SessionAttributesKeyT = Aws::String, typename SessionAttributesValueT = Aws::String>
  PostTextRequest& AddSessionAttributes(SessionAttributesKeyT&& key, SessionAttributesValueT&& value) {
    m_sessionAttributesHasBeenSet = true;
    m_sessionAttributes.emplace(std::forward<SessionAttributesKeyT>(key), std::forward<SessionAttributesValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Request-specific information passed between Amazon Lex and a client
   * application.</p> <p>The namespace <code>x-amz-lex:</code> is reserved for
   * special attributes. Don't create any request attributes with the prefix
   * <code>x-amz-lex:</code>.</p> <p>For more information, see <a
   * href="https://docs.aws.amazon.com/lex/latest/dg/context-mgmt.html#context-mgmt-request-attribs">Setting
   * Request Attributes</a>.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetRequestAttributes() const { return m_requestAttributes; }
  inline bool RequestAttributesHasBeenSet() const { return m_requestAttributesHasBeenSet; }
  template <typename RequestAttributesT = Aws::Map<Aws::String, Aws::String>>
  void SetRequestAttributes(RequestAttributesT&& value) {
    m_requestAttributesHasBeenSet = true;
    m_requestAttributes = std::forward<RequestAttributesT>(value);
  }
  template <typename RequestAttributesT = Aws::Map<Aws::String, Aws::String>>
  PostTextRequest& WithRequestAttributes(RequestAttributesT&& value) {
    SetRequestAttributes(std::forward<RequestAttributesT>(value));
    return *this;
  }
  template <typename RequestAttributesKeyT = Aws::String, typename RequestAttributesValueT = Aws::String>
  PostTextRequest& AddRequestAttributes(RequestAttributesKeyT&& key, RequestAttributesValueT&& value) {
    m_requestAttributesHasBeenSet = true;
    m_requestAttributes.emplace(std::forward<RequestAttributesKeyT>(key), std::forward<RequestAttributesValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The text that the user entered (Amazon Lex interprets this text).</p>
   */
  inline const Aws::String& GetInputText() const { return m_inputText; }
  inline bool InputTextHasBeenSet() const { return m_inputTextHasBeenSet; }
  template <typename InputTextT = Aws::String>
  void SetInputText(InputTextT&& value) {
    m_inputTextHasBeenSet = true;
    m_inputText = std::forward<InputTextT>(value);
  }
  template <typename InputTextT = Aws::String>
  PostTextRequest& WithInputText(InputTextT&& value) {
    SetInputText(std::forward<InputTextT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of contexts active for the request. A context can be activated when a
   * previous intent is fulfilled, or by including the context in the request,</p>
   * <p>If you don't specify a list of contexts, Amazon Lex will use the current list
   * of contexts for the session. If you specify an empty list, all contexts for the
   * session are cleared.</p>
   */
  inline const Aws::Vector<ActiveContext>& GetActiveContexts() const { return m_activeContexts; }
  inline bool ActiveContextsHasBeenSet() const { return m_activeContextsHasBeenSet; }
  template <typename ActiveContextsT = Aws::Vector<ActiveContext>>
  void SetActiveContexts(ActiveContextsT&& value) {
    m_activeContextsHasBeenSet = true;
    m_activeContexts = std::forward<ActiveContextsT>(value);
  }
  template <typename ActiveContextsT = Aws::Vector<ActiveContext>>
  PostTextRequest& WithActiveContexts(ActiveContextsT&& value) {
    SetActiveContexts(std::forward<ActiveContextsT>(value));
    return *this;
  }
  template <typename ActiveContextsT = ActiveContext>
  PostTextRequest& AddActiveContexts(ActiveContextsT&& value) {
    m_activeContextsHasBeenSet = true;
    m_activeContexts.emplace_back(std::forward<ActiveContextsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_botName;
  bool m_botNameHasBeenSet = false;

  Aws::String m_botAlias;
  bool m_botAliasHasBeenSet = false;

  Aws::String m_userId;
  bool m_userIdHasBeenSet = false;

  Aws::Map<Aws::String, Aws::String> m_sessionAttributes;
  bool m_sessionAttributesHasBeenSet = false;

  Aws::Map<Aws::String, Aws::String> m_requestAttributes;
  bool m_requestAttributesHasBeenSet = false;

  Aws::String m_inputText;
  bool m_inputTextHasBeenSet = false;

  Aws::Vector<ActiveContext> m_activeContexts;
  bool m_activeContextsHasBeenSet = false;
};

}  // namespace Model
}  // namespace LexRuntimeService
}  // namespace Aws
