﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/qconnect/QConnectRequest.h>
#include <aws/qconnect/QConnect_EXPORTS.h>
#include <aws/qconnect/model/ChannelSubtype.h>
#include <aws/qconnect/model/GroupingConfiguration.h>
#include <aws/qconnect/model/MessageTemplateAttributes.h>
#include <aws/qconnect/model/MessageTemplateContentProvider.h>
#include <aws/qconnect/model/MessageTemplateSourceConfiguration.h>

#include <utility>

namespace Aws {
namespace QConnect {
namespace Model {

/**
 */
class CreateMessageTemplateRequest : public QConnectRequest {
 public:
  AWS_QCONNECT_API CreateMessageTemplateRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateMessageTemplate"; }

  AWS_QCONNECT_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The identifier of the knowledge base. Can be either the ID or the ARN. URLs
   * cannot contain the ARN.</p>
   */
  inline const Aws::String& GetKnowledgeBaseId() const { return m_knowledgeBaseId; }
  inline bool KnowledgeBaseIdHasBeenSet() const { return m_knowledgeBaseIdHasBeenSet; }
  template <typename KnowledgeBaseIdT = Aws::String>
  void SetKnowledgeBaseId(KnowledgeBaseIdT&& value) {
    m_knowledgeBaseIdHasBeenSet = true;
    m_knowledgeBaseId = std::forward<KnowledgeBaseIdT>(value);
  }
  template <typename KnowledgeBaseIdT = Aws::String>
  CreateMessageTemplateRequest& WithKnowledgeBaseId(KnowledgeBaseIdT&& value) {
    SetKnowledgeBaseId(std::forward<KnowledgeBaseIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the message template.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateMessageTemplateRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The content of the message template.</p>
   */
  inline const MessageTemplateContentProvider& GetContent() const { return m_content; }
  inline bool ContentHasBeenSet() const { return m_contentHasBeenSet; }
  template <typename ContentT = MessageTemplateContentProvider>
  void SetContent(ContentT&& value) {
    m_contentHasBeenSet = true;
    m_content = std::forward<ContentT>(value);
  }
  template <typename ContentT = MessageTemplateContentProvider>
  CreateMessageTemplateRequest& WithContent(ContentT&& value) {
    SetContent(std::forward<ContentT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The description of the message template.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  CreateMessageTemplateRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The channel subtype this message template applies to.</p>
   */
  inline ChannelSubtype GetChannelSubtype() const { return m_channelSubtype; }
  inline bool ChannelSubtypeHasBeenSet() const { return m_channelSubtypeHasBeenSet; }
  inline void SetChannelSubtype(ChannelSubtype value) {
    m_channelSubtypeHasBeenSet = true;
    m_channelSubtype = value;
  }
  inline CreateMessageTemplateRequest& WithChannelSubtype(ChannelSubtype value) {
    SetChannelSubtype(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The language code value for the language in which the quick response is
   * written. The supported language codes include <code>de_DE</code>,
   * <code>en_US</code>, <code>es_ES</code>, <code>fr_FR</code>, <code>id_ID</code>,
   * <code>it_IT</code>, <code>ja_JP</code>, <code>ko_KR</code>, <code>pt_BR</code>,
   * <code>zh_CN</code>, <code>zh_TW</code> </p>
   */
  inline const Aws::String& GetLanguage() const { return m_language; }
  inline bool LanguageHasBeenSet() const { return m_languageHasBeenSet; }
  template <typename LanguageT = Aws::String>
  void SetLanguage(LanguageT&& value) {
    m_languageHasBeenSet = true;
    m_language = std::forward<LanguageT>(value);
  }
  template <typename LanguageT = Aws::String>
  CreateMessageTemplateRequest& WithLanguage(LanguageT&& value) {
    SetLanguage(std::forward<LanguageT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The source configuration of the message template. Only set this argument for
   * WHATSAPP channel subtype.</p>
   */
  inline const MessageTemplateSourceConfiguration& GetSourceConfiguration() const { return m_sourceConfiguration; }
  inline bool SourceConfigurationHasBeenSet() const { return m_sourceConfigurationHasBeenSet; }
  template <typename SourceConfigurationT = MessageTemplateSourceConfiguration>
  void SetSourceConfiguration(SourceConfigurationT&& value) {
    m_sourceConfigurationHasBeenSet = true;
    m_sourceConfiguration = std::forward<SourceConfigurationT>(value);
  }
  template <typename SourceConfigurationT = MessageTemplateSourceConfiguration>
  CreateMessageTemplateRequest& WithSourceConfiguration(SourceConfigurationT&& value) {
    SetSourceConfiguration(std::forward<SourceConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An object that specifies the default values to use for variables in the
   * message template. This object contains different categories of key-value pairs.
   * Each key defines a variable or placeholder in the message template. The
   * corresponding value defines the default value for that variable.</p>
   */
  inline const MessageTemplateAttributes& GetDefaultAttributes() const { return m_defaultAttributes; }
  inline bool DefaultAttributesHasBeenSet() const { return m_defaultAttributesHasBeenSet; }
  template <typename DefaultAttributesT = MessageTemplateAttributes>
  void SetDefaultAttributes(DefaultAttributesT&& value) {
    m_defaultAttributesHasBeenSet = true;
    m_defaultAttributes = std::forward<DefaultAttributesT>(value);
  }
  template <typename DefaultAttributesT = MessageTemplateAttributes>
  CreateMessageTemplateRequest& WithDefaultAttributes(DefaultAttributesT&& value) {
    SetDefaultAttributes(std::forward<DefaultAttributesT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const GroupingConfiguration& GetGroupingConfiguration() const { return m_groupingConfiguration; }
  inline bool GroupingConfigurationHasBeenSet() const { return m_groupingConfigurationHasBeenSet; }
  template <typename GroupingConfigurationT = GroupingConfiguration>
  void SetGroupingConfiguration(GroupingConfigurationT&& value) {
    m_groupingConfigurationHasBeenSet = true;
    m_groupingConfiguration = std::forward<GroupingConfigurationT>(value);
  }
  template <typename GroupingConfigurationT = GroupingConfiguration>
  CreateMessageTemplateRequest& WithGroupingConfiguration(GroupingConfigurationT&& value) {
    SetGroupingConfiguration(std::forward<GroupingConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A unique, case-sensitive identifier that you provide to ensure the
   * idempotency of the request. If not provided, the Amazon Web Services SDK
   * populates this field. For more information about idempotency, see <a
   * href="http://aws.amazon.com/builders-library/making-retries-safe-with-idempotent-APIs/">Making
   * retries safe with idempotent APIs</a>.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  CreateMessageTemplateRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tags used to organize, track, or control access for this resource.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  CreateMessageTemplateRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  CreateMessageTemplateRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_knowledgeBaseId;
  bool m_knowledgeBaseIdHasBeenSet = false;

  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  MessageTemplateContentProvider m_content;
  bool m_contentHasBeenSet = false;

  Aws::String m_description;
  bool m_descriptionHasBeenSet = false;

  ChannelSubtype m_channelSubtype{ChannelSubtype::NOT_SET};
  bool m_channelSubtypeHasBeenSet = false;

  Aws::String m_language;
  bool m_languageHasBeenSet = false;

  MessageTemplateSourceConfiguration m_sourceConfiguration;
  bool m_sourceConfigurationHasBeenSet = false;

  MessageTemplateAttributes m_defaultAttributes;
  bool m_defaultAttributesHasBeenSet = false;

  GroupingConfiguration m_groupingConfiguration;
  bool m_groupingConfigurationHasBeenSet = false;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_clientTokenHasBeenSet = true;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace QConnect
}  // namespace Aws
