﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/ram/RAMRequest.h>
#include <aws/ram/RAM_EXPORTS.h>

#include <utility>

namespace Aws {
namespace RAM {
namespace Model {

/**
 */
class DisassociateResourceSharePermissionRequest : public RAMRequest {
 public:
  AWS_RAM_API DisassociateResourceSharePermissionRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DisassociateResourceSharePermission"; }

  AWS_RAM_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The <a
   * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon
   * Resource Name (ARN)</a> of the resource share that you want to remove the
   * managed permission from.</p>
   */
  inline const Aws::String& GetResourceShareArn() const { return m_resourceShareArn; }
  inline bool ResourceShareArnHasBeenSet() const { return m_resourceShareArnHasBeenSet; }
  template <typename ResourceShareArnT = Aws::String>
  void SetResourceShareArn(ResourceShareArnT&& value) {
    m_resourceShareArnHasBeenSet = true;
    m_resourceShareArn = std::forward<ResourceShareArnT>(value);
  }
  template <typename ResourceShareArnT = Aws::String>
  DisassociateResourceSharePermissionRequest& WithResourceShareArn(ResourceShareArnT&& value) {
    SetResourceShareArn(std::forward<ResourceShareArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The <a
   * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon
   * Resource Name (ARN)</a> of the managed permission to disassociate from the
   * resource share. Changes to permissions take effect immediately.</p>
   */
  inline const Aws::String& GetPermissionArn() const { return m_permissionArn; }
  inline bool PermissionArnHasBeenSet() const { return m_permissionArnHasBeenSet; }
  template <typename PermissionArnT = Aws::String>
  void SetPermissionArn(PermissionArnT&& value) {
    m_permissionArnHasBeenSet = true;
    m_permissionArn = std::forward<PermissionArnT>(value);
  }
  template <typename PermissionArnT = Aws::String>
  DisassociateResourceSharePermissionRequest& WithPermissionArn(PermissionArnT&& value) {
    SetPermissionArn(std::forward<PermissionArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies a unique, case-sensitive identifier that you provide to ensure the
   * idempotency of the request. This lets you safely retry the request without
   * accidentally performing the same operation a second time. Passing the same value
   * to a later call to an operation requires that you also pass the same value for
   * all other parameters. We recommend that you use a <a
   * href="https://wikipedia.org/wiki/Universally_unique_identifier">UUID type of
   * value.</a>.</p> <p>If you don't provide this value, then Amazon Web Services
   * generates a random one for you.</p> <p>If you retry the operation with the same
   * <code>ClientToken</code>, but with different parameters, the retry fails with an
   * <code>IdempotentParameterMismatch</code> error.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  DisassociateResourceSharePermissionRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_resourceShareArn;
  bool m_resourceShareArnHasBeenSet = false;

  Aws::String m_permissionArn;
  bool m_permissionArnHasBeenSet = false;

  Aws::String m_clientToken;
  bool m_clientTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace RAM
}  // namespace Aws
