﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/wellarchitected/WellArchitectedRequest.h>
#include <aws/wellarchitected/WellArchitected_EXPORTS.h>
#include <aws/wellarchitected/model/LensStatusType.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace WellArchitected {
namespace Model {

/**
 */
class DeleteLensRequest : public WellArchitectedRequest {
 public:
  AWS_WELLARCHITECTED_API DeleteLensRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DeleteLens"; }

  AWS_WELLARCHITECTED_API Aws::String SerializePayload() const override;

  AWS_WELLARCHITECTED_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{

  inline const Aws::String& GetLensAlias() const { return m_lensAlias; }
  inline bool LensAliasHasBeenSet() const { return m_lensAliasHasBeenSet; }
  template <typename LensAliasT = Aws::String>
  void SetLensAlias(LensAliasT&& value) {
    m_lensAliasHasBeenSet = true;
    m_lensAlias = std::forward<LensAliasT>(value);
  }
  template <typename LensAliasT = Aws::String>
  DeleteLensRequest& WithLensAlias(LensAliasT&& value) {
    SetLensAlias(std::forward<LensAliasT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const Aws::String& GetClientRequestToken() const { return m_clientRequestToken; }
  inline bool ClientRequestTokenHasBeenSet() const { return m_clientRequestTokenHasBeenSet; }
  template <typename ClientRequestTokenT = Aws::String>
  void SetClientRequestToken(ClientRequestTokenT&& value) {
    m_clientRequestTokenHasBeenSet = true;
    m_clientRequestToken = std::forward<ClientRequestTokenT>(value);
  }
  template <typename ClientRequestTokenT = Aws::String>
  DeleteLensRequest& WithClientRequestToken(ClientRequestTokenT&& value) {
    SetClientRequestToken(std::forward<ClientRequestTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The status of the lens to be deleted.</p>
   */
  inline LensStatusType GetLensStatus() const { return m_lensStatus; }
  inline bool LensStatusHasBeenSet() const { return m_lensStatusHasBeenSet; }
  inline void SetLensStatus(LensStatusType value) {
    m_lensStatusHasBeenSet = true;
    m_lensStatus = value;
  }
  inline DeleteLensRequest& WithLensStatus(LensStatusType value) {
    SetLensStatus(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_lensAlias;
  bool m_lensAliasHasBeenSet = false;

  Aws::String m_clientRequestToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_clientRequestTokenHasBeenSet = true;

  LensStatusType m_lensStatus{LensStatusType::NOT_SET};
  bool m_lensStatusHasBeenSet = false;
};

}  // namespace Model
}  // namespace WellArchitected
}  // namespace Aws
