# frozen_string_literal: true

require 'json'
require 'stringio'
require 'zlib'
require 'active_support/core_ext/object/blank'

module GitlabQuality
  module TestTooling
    module CodeCoverage
      class Artifacts
        # Loads coverage artifacts from the filesystem
        #
        # @param test_reports [String] Glob pattern(s) for test JSON report files (RSpec or Jest).
        #   Supports comma-separated patterns (e.g., "jest/**/*.json,rspec/**/*.json")
        # @param coverage_report [String] Path to the LCOV coverage report file (e.g., "coverage/lcov/gitlab.lcov")
        # @param test_map [String] Path to the test map file, gzipped or plain JSON (e.g., "crystalball/packed-mapping.json.gz")
        def initialize(coverage_report:, test_map:, test_reports:)
          raise ArgumentError, "test_reports cannot be blank" if test_reports.blank?

          @test_reports_glob = test_reports
          @coverage_report_path = coverage_report
          @test_map_path = test_map
        end

        # Loads and parses test JSON report files (RSpec or Jest)
        #
        # @return [Array<Hash>] Array of parsed JSON test reports
        # @raise [RuntimeError] If no test reports are found or if JSON parsing fails
        def test_reports
          @test_report_files ||= test_reports_paths.map do |report_path|
            JSON.parse(File.read(report_path))
          rescue JSON::ParserError => e
            raise "Invalid JSON in test report file #{report_path}: #{e.message}"
          end
        end

        # Loads the LCOV coverage report file
        #
        # @return [String] Raw content of the LCOV coverage report
        # @raise [RuntimeError] If the coverage report file is not found
        def coverage_report
          @coverage_report ||= read_coverage_reports
        end

        # Loads and parses the test map file (supports gzipped or plain JSON)
        #
        # @return [Hash] Parsed test map data
        # @raise [RuntimeError] If the test map file is not found or cannot be parsed
        def test_map
          @test_map ||= fetch_test_map
        end

        private

        def test_reports_paths
          @test_reports_paths ||= begin
            # Support comma-separated glob patterns (e.g., "jest/**/*.json,rspec/**/*.json")
            patterns = @test_reports_glob.split(',').map(&:strip).reject(&:empty?)
            paths = Dir.glob(patterns)

            raise "No test reports found matching pattern: #{@test_reports_glob}" if paths.empty?

            paths
          end
        end

        def read_coverage_reports
          raise "Coverage report not found in: #{@coverage_report_path}" unless File.exist?(@coverage_report_path)

          File.read(@coverage_report_path)
        end

        def fetch_test_map
          raise "Test map file not found: #{@test_map_path}" unless File.exist?(@test_map_path)

          begin
            content = File.read(@test_map_path)

            # If it's a gzipped file, decompress it
            content = decompressed_gzip(content) if @test_map_path.end_with?('.gz')

            JSON.parse(content)
          rescue StandardError => e
            raise "Failed to read test map from #{@test_map_path}: #{e.message}"
          end
        end

        def decompressed_gzip(gzipped_data)
          Zlib::GzipReader.new(StringIO.new(gzipped_data)).read
        end
      end
    end
  end
end
