/*
 * Decompiled with CFR 0.152.
 */
package ghidra.program.util;

import db.Transaction;
import ghidra.framework.model.DomainFile;
import ghidra.framework.model.DomainObject;
import ghidra.framework.model.ProjectData;
import ghidra.framework.options.Options;
import ghidra.program.model.listing.CircularDependencyException;
import ghidra.program.model.listing.Library;
import ghidra.program.model.listing.Program;
import ghidra.program.model.symbol.ExternalLocation;
import ghidra.program.model.symbol.ExternalManager;
import ghidra.program.model.symbol.Namespace;
import ghidra.program.model.symbol.SourceType;
import ghidra.program.model.symbol.Symbol;
import ghidra.program.model.symbol.SymbolTable;
import ghidra.program.model.symbol.SymbolType;
import ghidra.util.Msg;
import ghidra.util.StringUtilities;
import ghidra.util.exception.CancelledException;
import ghidra.util.exception.DuplicateNameException;
import ghidra.util.exception.InvalidInputException;
import ghidra.util.exception.VersionException;
import ghidra.util.task.TaskMonitor;
import java.io.Closeable;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.TreeMap;
import java.util.function.Consumer;

public class ExternalSymbolResolver
implements Closeable {
    private static final String REQUIRED_LIBRARY_PROPERTY_PREFIX = "Required Library [";
    private final ProjectData projectData;
    private final TaskMonitor monitor;
    private final List<ProgramSymbolResolver> programsToFix = new ArrayList<ProgramSymbolResolver>();
    private final Map<String, Program> loadedPrograms = new HashMap<String, Program>();
    private final Map<String, Throwable> problemLibraries = new HashMap<String, Throwable>();

    public static String getRequiredLibraryProperty(int libraryIndex) {
        return String.format("%s %s]", REQUIRED_LIBRARY_PROPERTY_PREFIX, StringUtilities.pad((String)("" + libraryIndex), (char)' ', (int)4));
    }

    public static List<String> getOrderedRequiredLibraryNames(Program program) {
        TreeMap<Integer, String> orderLibraryMap = new TreeMap<Integer, String>();
        Options options = program.getOptions("Program Information");
        for (String optionName : options.getOptionNames()) {
            String libName;
            int prefixIndex = optionName.indexOf(REQUIRED_LIBRARY_PROPERTY_PREFIX);
            if (prefixIndex == -1 || !optionName.endsWith("]") || (libName = options.getString(optionName, null)) == null) continue;
            String indexStr = optionName.substring(prefixIndex + REQUIRED_LIBRARY_PROPERTY_PREFIX.length(), optionName.length() - 1).trim();
            try {
                orderLibraryMap.put(Integer.parseInt(indexStr), libName.trim());
            }
            catch (NumberFormatException e) {
                Msg.error(ExternalSymbolResolver.class, (Object)("Program contains invalid property: " + optionName));
            }
        }
        return new ArrayList<String>(orderLibraryMap.values());
    }

    public ExternalSymbolResolver(ProjectData projectData, TaskMonitor monitor) {
        this.projectData = projectData;
        this.monitor = monitor;
    }

    public void addProgramToFixup(Program program) {
        this.addProgramToFixup(program.getDomainFile().getPathname(), program);
    }

    public void addProgramToFixup(String programPath, Program program) {
        this.programsToFix.add(new ProgramSymbolResolver(program, programPath));
        this.addLoadedProgram(programPath, program);
    }

    public void addLoadedProgram(String programPath, Program program) {
        if (this.loadedPrograms.put(programPath, program) == null) {
            program.addConsumer((Object)this);
        }
    }

    public boolean hasProblemLibraries() {
        return !this.problemLibraries.isEmpty();
    }

    @Override
    public void close() {
        for (Program prog : this.loadedPrograms.values()) {
            prog.release((Object)this);
        }
        this.programsToFix.clear();
        this.loadedPrograms.clear();
    }

    public void fixUnresolvedExternalSymbols() throws CancelledException {
        for (ProgramSymbolResolver psr : this.programsToFix) {
            psr.resolveExternalSymbols();
        }
    }

    public void logInfo(Consumer<String> logger, boolean shortSummary) {
        for (ProgramSymbolResolver psr : this.programsToFix) {
            psr.log(logger, shortSummary);
        }
    }

    protected Program getLibraryProgram(String libPath) throws CancelledException {
        Program result = this.loadedPrograms.get(libPath);
        if (result == null && this.projectData != null && !this.problemLibraries.containsKey(libPath) && (result = this.openLibraryFile(this.projectData.getFile(libPath), libPath)) != null) {
            this.loadedPrograms.put(libPath, result);
        }
        return result;
    }

    protected Program openLibraryFile(DomainFile libDf, String libPath) throws CancelledException {
        try {
            if (libDf == null) {
                throw new IOException("Dangling external path: " + libPath);
            }
            DomainObject libDo = libDf.getDomainObject((Object)this, false, false, this.monitor);
            if (libDo instanceof Program) {
                Program p = (Program)libDo;
                return p;
            }
            libDo.release((Object)this);
            throw new IOException("Referenced external program is not a program: " + libPath);
        }
        catch (VersionException | IOException e) {
            this.problemLibraries.put(libPath, e);
            return null;
        }
    }

    private static boolean isExportedSymbol(Program program, String name) {
        for (Symbol s : program.getSymbolTable().getLabelOrFunctionSymbols(name, null)) {
            if (!s.isExternalEntryPoint()) continue;
            return true;
        }
        return false;
    }

    private class ProgramSymbolResolver {
        Program program;
        String programPath;
        int externalSymbolCount;
        List<Long> unresolvedExternalFunctionIds;
        List<ExtLibInfo> extLibs = new ArrayList<ExtLibInfo>();

        private ProgramSymbolResolver(Program program, String programPath) {
            this.program = program;
            this.programPath = programPath;
        }

        private int getResolvedSymbolCount() {
            return this.externalSymbolCount - this.unresolvedExternalFunctionIds.size();
        }

        private void log(Consumer<String> logger, boolean shortSummary) {
            boolean changed;
            boolean bl = changed = this.unresolvedExternalFunctionIds.size() != this.externalSymbolCount;
            if (this.extLibs.isEmpty() && this.externalSymbolCount == 0) {
                return;
            }
            if (!changed && !this.hasSomeLibrariesConfigured()) {
                logger.accept("Resolving External Symbols of [%s] - %d unresolved symbols, no external libraries configured - skipping".formatted(this.programPath, this.externalSymbolCount));
                return;
            }
            logger.accept("Resolving External Symbols of [%s]%s".formatted(this.programPath, shortSummary ? " - Summary" : ""));
            logger.accept("\t%d external symbols resolved, %d remain unresolved".formatted(this.getResolvedSymbolCount(), this.unresolvedExternalFunctionIds.size()));
            for (ExtLibInfo extLib : this.extLibs) {
                if (extLib.problem != null) {
                    logger.accept("\t[%s] -> %s, %s".formatted(extLib.name, extLib.getLibPath(), extLib.getProblemMessage()));
                } else if (extLib.programPath != null) {
                    logger.accept("\t[%s] -> %s, %d new symbols resolved".formatted(extLib.name, extLib.getLibPath(), extLib.resolvedSymbols.size()));
                } else {
                    logger.accept("\t[%s] -> %s".formatted(extLib.name, extLib.getLibPath()));
                }
                if (shortSummary) continue;
                for (String symbolName : extLib.resolvedSymbols) {
                    logger.accept("\t\t[%s]".formatted(symbolName));
                }
            }
            if (!shortSummary && changed && !this.unresolvedExternalFunctionIds.isEmpty()) {
                logger.accept("\tUnresolved remaining %d:".formatted(this.unresolvedExternalFunctionIds.size()));
                SymbolTable symbolTable = this.program.getSymbolTable();
                for (Long symId : this.unresolvedExternalFunctionIds) {
                    Symbol s = symbolTable.getSymbol(symId.longValue());
                    logger.accept("\t\t[%s]".formatted(s.getName()));
                }
            }
        }

        private boolean hasSomeLibrariesConfigured() {
            for (ExtLibInfo extLib : this.extLibs) {
                if (extLib.program == null && extLib.problem == null && extLib.programPath == null) continue;
                return true;
            }
            return false;
        }

        private void resolveExternalSymbols() throws CancelledException {
            this.unresolvedExternalFunctionIds = this.getUnresolvedExternalFunctionIds();
            this.externalSymbolCount = this.unresolvedExternalFunctionIds.size();
            if (this.unresolvedExternalFunctionIds.isEmpty()) {
                return;
            }
            this.extLibs = this.getLibsToSearch();
            if (!this.extLibs.isEmpty()) {
                try (Transaction tx = this.program.openTransaction("Resolve External Symbols");){
                    for (ExtLibInfo extLib : this.extLibs) {
                        ExternalSymbolResolver.this.monitor.checkCancelled();
                        this.resolveSymbolsToLibrary(extLib);
                    }
                }
            }
        }

        private List<ExtLibInfo> getLibsToSearch() throws CancelledException {
            ArrayList<ExtLibInfo> result = new ArrayList<ExtLibInfo>();
            ExternalManager externalManager = this.program.getExternalManager();
            for (String libName : ExternalSymbolResolver.getOrderedRequiredLibraryNames(this.program)) {
                Library lib = externalManager.getExternalLibrary(libName);
                String libPath = lib != null ? lib.getAssociatedProgramPath() : null;
                Program libProg = libPath != null ? ExternalSymbolResolver.this.getLibraryProgram(libPath) : null;
                Throwable problem = libProg == null && libPath != null ? ExternalSymbolResolver.this.problemLibraries.get(libPath) : null;
                result.add(new ExtLibInfo(libName, lib, libPath, libProg, new ArrayList<String>(), problem));
            }
            return result;
        }

        private void resolveSymbolsToLibrary(ExtLibInfo extLib) throws CancelledException {
            if (extLib.program == null) {
                return;
            }
            ExternalManager externalManager = this.program.getExternalManager();
            SymbolTable symbolTable = this.program.getSymbolTable();
            Iterator<Long> idIterator = this.unresolvedExternalFunctionIds.iterator();
            while (idIterator.hasNext()) {
                ExternalSymbolResolver.this.monitor.checkCancelled();
                Symbol s = symbolTable.getSymbol(idIterator.next().longValue());
                if (s == null || !s.isExternal() || s.getSymbolType() != SymbolType.FUNCTION) {
                    Msg.error(ExternalSymbolResolver.class, (Object)"Concurrent modification of symbol table while resolving external symbols");
                    idIterator.remove();
                    continue;
                }
                ExternalLocation extLoc = externalManager.getExternalLocation(s);
                String extLocName = Objects.requireNonNullElse(extLoc.getOriginalImportedName(), extLoc.getLabel());
                if (!ExternalSymbolResolver.isExportedSymbol(extLib.program, extLocName)) continue;
                try {
                    s.setNamespace((Namespace)extLib.lib);
                    idIterator.remove();
                    extLib.resolvedSymbols.add(s.getName());
                }
                catch (CircularDependencyException | DuplicateNameException | InvalidInputException e) {
                    Msg.error(ExternalSymbolResolver.class, (Object)("Error setting external symbol namespace for " + extLoc.getLabel()), (Throwable)e);
                }
            }
        }

        private List<Long> getUnresolvedExternalFunctionIds() {
            ArrayList<Long> symbolIds = new ArrayList<Long>();
            ExternalManager externalManager = this.program.getExternalManager();
            Library library = externalManager.getExternalLibrary("<EXTERNAL>");
            if (library != null) {
                for (Symbol s : this.program.getSymbolTable().getSymbols((Namespace)library)) {
                    if (s.getSymbolType() != SymbolType.FUNCTION || s.getSource() == SourceType.DEFAULT) continue;
                    symbolIds.add(s.getID());
                }
            }
            return symbolIds;
        }

        record ExtLibInfo(String name, Library lib, String programPath, Program program, List<String> resolvedSymbols, Throwable problem) {
            String getProblemMessage() {
                Throwable throwable = this.problem;
                if (throwable instanceof VersionException) {
                    VersionException ve = (VersionException)throwable;
                    return this.getVersionError(ve);
                }
                return this.problem != null ? this.problem.getMessage() : "";
            }

            String getLibPath() {
                return this.programPath != null ? this.programPath : "missing";
            }

            String getVersionError(VersionException ve) {
                String versionType = switch (ve.getVersionIndicator()) {
                    case 2 -> " newer";
                    case 1 -> "n older";
                    default -> "n unknown";
                };
                String upgradeMsg = ve.isUpgradable() ? " (upgrade is possible)" : "";
                return "skipped: file was created with a%s version of Ghidra%s".formatted(versionType, upgradeMsg);
            }
        }
    }
}

