/*
 * Decompiled with CFR 0.152.
 */
package ghidra.util.timer;

import ghidra.util.timer.GTimer;
import ghidra.util.timer.GTimerMonitor;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

public class GTimerCache<K, V> {
    private static final int INITIAL_MAP_SIZE = 16;
    private static final float LOAD_FACTOR = 0.75f;
    private int capacity;
    private long lifetime;
    private Runnable timerExpiredRunnable = this::timerExpired;
    private Map<K, CachedValue> map;
    private GTimerMonitor timerMonitor;

    public GTimerCache(Duration lifetime, int capacity) {
        if (lifetime.isZero() || lifetime.isNegative()) {
            throw new IllegalArgumentException("The duration must be a time > 0!");
        }
        if (capacity < 1) {
            throw new IllegalArgumentException("The capacity must be > 0!");
        }
        this.lifetime = lifetime.toMillis();
        this.capacity = capacity;
        this.map = new LinkedHashMap<K, CachedValue>(16, 0.75f, true){

            @Override
            protected boolean removeEldestEntry(Map.Entry<K, CachedValue> eldest) {
                if (this.size() > GTimerCache.this.capacity) {
                    GTimerCache.this.valueRemoved(eldest.getKey(), eldest.getValue().getValue());
                    return true;
                }
                return false;
            }
        };
    }

    public synchronized void setCapacity(int capacity) {
        if (capacity < 1) {
            throw new IllegalArgumentException("The capacity must be > 0!");
        }
        this.capacity = capacity;
        if (this.map.size() <= capacity) {
            return;
        }
        Iterator<Map.Entry<K, CachedValue>> it = this.map.entrySet().iterator();
        int n = this.map.size() - capacity;
        for (int i = 0; i < n; ++i) {
            Map.Entry<K, CachedValue> next = it.next();
            it.remove();
            CachedValue value = next.getValue();
            this.valueRemoved(value.getKey(), value.getValue());
        }
    }

    public synchronized void setDuration(Duration duration) {
        if (duration.isZero() || duration.isNegative()) {
            throw new IllegalArgumentException("The duration must be a time > 0!");
        }
        this.lifetime = duration.toMillis();
        if (this.timerMonitor != null) {
            this.timerMonitor.cancel();
            this.timerMonitor = null;
        }
        this.timerExpired();
    }

    public synchronized V put(K key, V value) {
        V previous;
        Objects.requireNonNull(key);
        Objects.requireNonNull(value);
        CachedValue old = this.map.put(key, new CachedValue(key, value));
        V v = previous = old == null ? null : (V)old.getValue();
        if (!Objects.equals(value, previous)) {
            if (previous != null) {
                this.valueRemoved(key, previous);
            }
            this.valueAdded(key, value);
        }
        if (this.timerMonitor == null) {
            this.timerMonitor = GTimer.scheduleRunnable(this.lifetime, this.timerExpiredRunnable);
        }
        return previous;
    }

    public synchronized V remove(K key) {
        CachedValue removed = this.map.remove(key);
        if (removed == null) {
            return null;
        }
        this.valueRemoved(removed.getKey(), removed.getValue());
        return removed.value;
    }

    public synchronized boolean containsKey(K key) {
        return this.map.containsKey(key);
    }

    public synchronized int size() {
        return this.map.size();
    }

    public synchronized V get(K key) {
        CachedValue cachedValue = this.map.get(key);
        if (cachedValue == null) {
            return null;
        }
        cachedValue.updateAccessTime();
        return cachedValue.getValue();
    }

    public synchronized void clear() {
        for (Map.Entry<K, CachedValue> entry : this.map.entrySet()) {
            CachedValue value = entry.getValue();
            this.valueRemoved(value.getKey(), value.getValue());
        }
        this.map.clear();
    }

    protected void valueRemoved(K key, V value) {
    }

    protected void valueAdded(K key, V value) {
    }

    protected boolean shouldRemoveFromCache(K key, V value) {
        return true;
    }

    private synchronized void timerExpired() {
        this.timerMonitor = null;
        long eventTime = System.currentTimeMillis();
        List<CachedValue> expiredValues = this.getAndRemoveExpiredValues(eventTime);
        this.purgeOrReinstateExpiredValues(expiredValues);
        this.restartTimer(eventTime);
    }

    private List<CachedValue> getAndRemoveExpiredValues(long eventTime) {
        CachedValue next;
        ArrayList<CachedValue> expiredValues = new ArrayList<CachedValue>();
        Iterator<CachedValue> it = this.map.values().iterator();
        while (it.hasNext() && (next = it.next()).isExpired(eventTime)) {
            expiredValues.add(next);
            it.remove();
        }
        return expiredValues;
    }

    private void purgeOrReinstateExpiredValues(List<CachedValue> expiredValues) {
        for (CachedValue cachedValue : expiredValues) {
            if (this.shouldRemoveFromCache(cachedValue.getKey(), cachedValue.getValue())) {
                this.valueRemoved(cachedValue.getKey(), cachedValue.getValue());
                continue;
            }
            cachedValue.updateAccessTime();
            this.map.put(cachedValue.getKey(), cachedValue);
        }
    }

    private void restartTimer(long eventTime) {
        if (this.map.isEmpty()) {
            return;
        }
        CachedValue first = this.map.values().iterator().next();
        long elapsed = eventTime - first.getLastAccessedTime();
        long remaining = this.lifetime - elapsed;
        this.timerMonitor = GTimer.scheduleRunnable(remaining, this.timerExpiredRunnable);
    }

    private class CachedValue {
        private final K key;
        private final V value;
        private long lastAccessedTime;

        CachedValue(K key, V value) {
            this.key = key;
            this.value = value;
            this.lastAccessedTime = System.currentTimeMillis();
        }

        void updateAccessTime() {
            this.lastAccessedTime = System.currentTimeMillis();
        }

        long getLastAccessedTime() {
            return this.lastAccessedTime;
        }

        K getKey() {
            return this.key;
        }

        V getValue() {
            return this.value;
        }

        boolean isExpired(long eventTime) {
            long elapsed = eventTime - this.lastAccessedTime;
            return elapsed >= GTimerCache.this.lifetime;
        }
    }
}

