/*
 * Decompiled with CFR 0.152.
 */
package ghidra.file.formats.cramfs;

import ghidra.app.util.bin.ByteProvider;
import ghidra.file.formats.cramfs.CramFsInode;
import ghidra.file.formats.zlib.ZLIB;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;

public class CramFsBlockReader {
    private ByteProvider provider;
    private CramFsInode cramfsInode;
    private List<Integer> blockPointerTable = new LinkedList<Integer>();
    private List<Integer> compressedBlockSizes = new ArrayList<Integer>();
    private boolean isLittleEndian;

    public CramFsBlockReader(ByteProvider provider, CramFsInode cramfsInode, boolean isLittleEndian) throws IOException {
        this.provider = provider;
        this.cramfsInode = cramfsInode;
        this.isLittleEndian = isLittleEndian;
        this.populateBlockPointerTable();
        this.calculateCompressedBlockSizes();
    }

    private void populateBlockPointerTable() throws IOException {
        int numBlockPointers = this.calculateStartAddress();
        if (numBlockPointers < 0) {
            throw new IOException("Start Address for data block not found");
        }
        int inodeDataOffset = this.cramfsInode.getOffsetAdjusted();
        for (int i = 0; i < numBlockPointers - 1; ++i) {
            byte[] tempBuffer = this.provider.readBytes((long)inodeDataOffset, 4L);
            ByteBuffer byteBuffer = ByteBuffer.wrap(tempBuffer);
            if (this.isLittleEndian) {
                this.blockPointerTable.add(Integer.reverseBytes(byteBuffer.getInt()));
            } else {
                this.blockPointerTable.add(byteBuffer.getInt());
            }
            inodeDataOffset += 4;
        }
    }

    private int calculateStartAddress() throws IOException {
        int dataOffset;
        int numBlockPointers = -1;
        int dataOffsetStart = dataOffset = this.cramfsInode.getOffsetAdjusted();
        boolean firstAddressFound = false;
        while (!firstAddressFound) {
            byte[] possibleZlibHeader = this.provider.readBytes((long)dataOffset, 2L);
            if (Arrays.equals(possibleZlibHeader, ZLIB.ZLIB_COMPRESSION_DEFAULT) || Arrays.equals(possibleZlibHeader, ZLIB.ZLIB_COMPRESSION_BEST) || Arrays.equals(possibleZlibHeader, ZLIB.ZLIB_COMPRESSION_NO_LOW)) {
                this.blockPointerTable.add(dataOffset);
                firstAddressFound = true;
                return (dataOffset - dataOffsetStart) / 4 + 1;
            }
            dataOffset += 4;
        }
        return numBlockPointers;
    }

    private void calculateCompressedBlockSizes() {
        for (int i = 0; i < this.blockPointerTable.size() - 1; ++i) {
            this.compressedBlockSizes.add(this.blockPointerTable.get(i + 1) - this.blockPointerTable.get(i));
        }
    }

    public byte[] readDataBlock(int dataBlockIndex) throws IOException {
        return this.provider.readBytes((long)this.blockPointerTable.get(dataBlockIndex).intValue(), (long)this.compressedBlockSizes.get(dataBlockIndex).intValue());
    }

    public InputStream readDataBlockDecompressed(int dataBlockIndex) throws IOException {
        Integer index = this.blockPointerTable.get(dataBlockIndex);
        Integer length = this.compressedBlockSizes.get(dataBlockIndex);
        byte[] compressedBytes = this.provider.readBytes((long)index.intValue(), (long)length.intValue());
        ByteArrayInputStream compressedInputStream = new ByteArrayInputStream(compressedBytes);
        ZLIB zlib = new ZLIB();
        ByteArrayOutputStream decompressedOutputStream = zlib.decompress(compressedInputStream, 4096);
        return new ByteArrayInputStream(decompressedOutputStream.toByteArray());
    }

    public ByteProvider getProvider() {
        return this.provider;
    }

    public CramFsInode getCramfsInode() {
        return this.cramfsInode;
    }

    public List<Integer> getBlockPointerTable() {
        return this.blockPointerTable;
    }

    public int getNumBlockPointers() {
        return this.blockPointerTable.size() - 1;
    }
}

