/* ###
 * IP: GHIDRA
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package ghidra.app.plugin.core.byteviewer;

import java.awt.Color;
import java.awt.FontMetrics;
import java.math.BigInteger;

import org.apache.commons.lang3.StringUtils;

import docking.widgets.fieldpanel.field.Field;
import docking.widgets.fieldpanel.field.SimpleTextField;
import docking.widgets.fieldpanel.support.FieldHighlightFactory;
import docking.widgets.fieldpanel.support.Highlight;
import ghidra.app.plugin.core.format.ByteBlockInfo;

/**
 * Implementation for the index/address field.
 */
class IndexFieldFactory {

	private FontMetrics metrics;
	private int width;
	private IndexMap indexMap;
	private int charWidth;
	private String noValueStr;
	private int startX;
	private Color missingValueColor;
	private FieldHighlightFactory highlightFactory = new DummyHighlightFactory();

	/**
	 * Constructor
	 * @param metrics the FontMetrics this field should use to do size computations
	 */
	IndexFieldFactory(FontMetrics metrics) {
		this.metrics = metrics;

		charWidth = metrics.charWidth('W');
		width = ByteViewerComponentProvider.DEFAULT_NUMBER_OF_CHARS * charWidth;
		missingValueColor = ByteViewerComponentProvider.SEPARATOR_COLOR;
		startX = charWidth;

		// initialize to a non-null value
		setSize(1);
	}

	/**
	 * Gets a Field object for the given index.
	 * @param index the index to get a Field for
	 * @return The Field for the given index
	 */
	public Field getField(BigInteger index) {

		if (indexMap == null) {
			return null;
		}

		// translate index to block and offset into the block
		ByteBlockInfo info = indexMap.getBlockInfo(index, 0);
		if (info == null) {
			for (int i = 0; i < indexMap.getBytesPerLine(); i++) {
				info = indexMap.getBlockInfo(index, i);
				if (info != null) {
					break;
				}
			}
			if (info == null) {
				if (indexMap.isBlockSeparatorIndex(index)) {
					SimpleTextField sf = new SimpleTextField(noValueStr, metrics, startX, width,
						false, highlightFactory);

					sf.setForeground(missingValueColor);
					return sf;
				}
				return null;
			}
		}
		String locRep = info.getBlock().getLocationRepresentation(info.getOffset());
		if (locRep == null) {
			return null;
		}
		return new SimpleTextField(locRep, metrics, startX, width, false, highlightFactory);
	}

	public FontMetrics getMetrics() {
		return metrics;
	}

	/**
	 * Sets the starting x position for the fields generated by this factory
	 * @param x The 
	 */
	public void setStartX(int x) {
		startX = x;
	}

	/**
	 * Returns the starting x position for the fields generated by this factory.
	 * @return the starting x position for the fields generated by this factory.
	 */
	public int getStartX() {
		return startX;
	}

	/**
	 * Returns the width of the fields associated with this Factory.
	 * @return  the width of the fields associated with this Factory
	 */
	public int getWidth() {
		return width;
	}

	/**
	 * Sets the indexMap which is the data source for the generated fields.
	 * @param indexMap indexMap which produces the index string for specific indexes
	 */
	void setIndexMap(IndexMap indexMap) {
		this.indexMap = indexMap;
	}

	/**
	 * Sets the size of this field based on the number of chars to display 
	 * @param charCount the number of chars to 
	 */
	void setSize(int charCount) {
		width = charCount * charWidth;
		// add in space for margins
		width += 2 * charWidth;

		noValueStr = StringUtils.repeat('.', charCount);
	}

	void setMissingValueColor(Color c) {
		missingValueColor = c;
	}

	static class DummyHighlightFactory implements FieldHighlightFactory {
		private final Highlight[] NO_HIGHLIGHTS = new Highlight[0];

		public DummyHighlightFactory() {
		}

		@Override
		public Highlight[] createHighlights(Field field, String text, int cursorTextOffset) {
			return NO_HIGHLIGHTS;
		}
	}

}
